#include "heap.h"
#include "scheduler.h"
#include "iostream.h"
#include "BasicStrategy.h"
#include "BasicFifoStrategy.h"
#include "ProfitStrategy.h"
#include "LazyScheduleStrategy.h"

extern double time;

#define MAX_Q_SIZE 1000
extern double responseTime;
extern double finishTime;

Scheduler::Scheduler(int nproc) { 
    this->nproc = nproc;
    num_free_proc = nproc;
    free_proc_vector = new char[nproc];
    memset(free_proc_vector, 1, nproc);

    strategy = new LazyScheduleStrategy(nproc);
    strategy = new ProfitStrategy(nproc);
    strategy->free_proc_vector = free_proc_vector;
    strategy->num_free_proc = num_free_proc;

    waitq = NULL;
    runq = NULL;

    previousEventTime = 0.0;
    utilization = 0.0;
    qsize = 0;
    nfinished = 0;
};

void Scheduler::updateUtil(){
    utilization = utilization * previousEventTime 
	+ ((time - previousEventTime)*((nproc - num_free_proc) * 100))/nproc;

    utilization = utilization / time;

    //    printf("Utilization = %5.3lf\n", utilization);

    cur_qsize = 0;
    Job *jptr = waitq;
    while(jptr != NULL){
	cur_qsize ++;
	jptr = jptr->next;
    }

    jptr = runq;
    while(jptr != NULL){
	cur_qsize ++;
	jptr = jptr->next;
    }
    
    qsize = qsize * previousEventTime + 
	+ (time - previousEventTime) * cur_qsize;
    qsize = qsize / time;

    previousEventTime = time;
}

void Scheduler::handleArrival(Event *e) {
    updateUtil();

    /*
    if(cur_qsize > MAX_Q_SIZE)
	return;
    */
    //    printf("%5.3lf: arrival \n", time);
    
    e->job = new SimJob(nproc);
    /*
    if(e->job->dbid == 6111)
	printf("debug dump 6111\n");
    */
    //Schedule job at the end of the queue.
    if(waitq){
	Job *jptr = waitq;
	while(jptr->next != NULL)
	    jptr = jptr->next;

	jptr->next = e->job;
    }
    else waitq = e->job;

    //    printf("MinPE = %d, MaxPE = %d, Work = %5.3lf\n", e->job->min_proc, e->job->max_proc, e->job->work);
    
    callStrategy();
}

void Scheduler::callStrategy(){

    //printf("callStrategy\n");
    
    Job *jptr = runq;
    while(jptr != NULL){
	((SimJob *)jptr)->updateWork();
	jptr = jptr->next;
    }
    
    strategy->num_free_proc = num_free_proc;
    strategy->allocate_processors(waitq, runq);
    
    //    printf("after allocate\n");
    num_free_proc = strategy->num_free_proc;
    
    jptr = runq;
    while(jptr != NULL){
	if(jptr->bitmap_changed)
	    ((SimJob *)jptr)->updateCompletionTime();
	jptr = jptr->next;
    }
    
    jptr = waitq;
    Job *prev = NULL;
    while(jptr != NULL){
	if((jptr->bitmap_changed) || (jptr->status == REJECTED)){

            // Assign waitq
            if(prev)
                prev->next = jptr->next;
            else
                waitq = jptr->next;
        
	    if(jptr->status != REJECTED){
		scheduleJob((SimJob *) jptr);	    
		// Assign runq          
		jptr->next = runq;
		runq = jptr;
	    }
            
            // Assign jptr
            if(prev)
                jptr = prev->next;
            else
                jptr = waitq;
        }
        else{
	    jptr->status = SCHEDULED;
            prev = jptr;
            jptr = jptr->next;
        }
    }
}

void Scheduler::handleJobDone(Event *e) {
    updateUtil();

    nfinished ++;
    /*
    if(nfinished > 18)
	printf("debug dump\n");
    */
    //    printf("%5.3lf: Job %d Finished\n", time, e->job->dbid);

    Job *jptr = runq;    
    Job *prev = NULL;
    
    while(jptr != NULL){
        if(jptr == (Job *)e->job){
	    
	    if((int)time > (int)(jptr->arrivalTime + jptr->deadline))
		printf("Deadline Missed by %d at %5.3lf with %5.3lf\n", jptr->dbid, time, jptr->arrivalTime + jptr->deadline);

	    num_free_proc += jptr->delete_all(free_proc_vector);
            
	    //Calculate response time, finish time.

            if(prev){
                prev->next = jptr->next;
		//    delete jptr;
                jptr = prev->next;
            }
            else{
                runq = jptr->next;
		//    delete jptr;
                jptr = runq;
            }           
	    break;
        }
        else{
            prev = jptr;
	    jptr = jptr->next;
        }
    }

    responseTime += e->job->startTime - e->job->arrivalTime;
    finishTime += time - e->job->arrivalTime;
    callStrategy();
}

void Scheduler::scheduleJob(SimJob *j) {
    //Start the job.

    j->startTime = time;
    j->previousEventTime = time;
    j->status = RUNNING;

    //Also inserts the event.
    j->updateCompletionTime();
}
