/* 
Starts the application

Orion Sky Lawlor, olawlor@acm.org, 11/28/2001
Updated by Isaac Dooley, 11/06/2005
*/
#include <string.h>
#include <stdlib.h>
#include "mytk.h"
#include "main.h"
#include "ccs-client.h"
#include "sockRoutines.h"


/*** Utility routines ***/

extern "C" void tcl_skt_idleFn(void) {
	Tcl_DoOneEvent(0);
}
extern "C" int tcl_skt_abortFn(int errCode,const char *msg) {
	Tcl_VarEval(interp,"fatalError \"Communication socket error: ",msg,"\"",NULL);
	return 0;
}
void bad(const char *why) {
	fprintf(stderr,"Fatal error: %s\n",why);
	if (interp) Tcl_VarEval(interp,"fatalError {",why,"}",NULL);
	fprintf(stderr,"Fatal error: %s\n",why);
	exit(1);
}
void status(const char *what) {
	printf("Now: %s\n",what);
}
static void chk(int ret) {
	if (ret==TCL_OK) return;
	Tcl_Eval(interp,"fatalTcl");
	exit(1);
}

/********** Mesh data interface **********/
class meshDataSource {
public:
	/// Return the number of processors
	virtual int getNumPe(void) =0;
	/// Return the number of chunks
	virtual int getNumChunks(void) =0;
	
	/// Update chunks:: with the next set of data
	virtual void update(void) =0;

  /// previous will go back one timestep if implemented
	virtual void previous(void)=0;

};

static meshDataSource *meshSource=NULL;

/************* CCS mesh source *****************/
class ccsMeshDataSource : public meshDataSource {
  CcsServer ccsServer;

public:
  ccsMeshDataSource(const char *host) {
	char *sep=strchr(host,':');
	if (sep==NULL) bad("You must separate the host and port with a colon");
	
	*sep++=0; //divide host name from port
	int port;
	if (1!=sscanf(sep,"%d",&port)) bad("Cannot parse port number");
	
//	printf("CCS connecting to '%s' port %d\n",host,port);
	CcsConnect(&ccsServer,host,port,NULL);
  }

  int getNumPe(void) {
	return CcsNumPes(&ccsServer);
  }
  int getNumChunks(void) {
	//FIXME: may have multiple chunks per PE
	return CcsNumPes(&ccsServer);
  }
  void update(void) {
	//Request the latest version from each chunk 
	// (should request them all in parallel, but this hangs
	//   charmrun (!) for large number of processors)
    for (int c=0;c<chunks::n;c++) {
      CcsServer &srv=ccsServer; 
      CcsSendRequest(&srv,"NetFEM_current",c,0,NULL);
      int size; void *buf;
      CcsRecvResponseMsg(&srv,&size,&buf,60);
      
      NetFEM_update *u=new NetFEM_update(0,0,0);
      unpack(*u,buf,size);
      free(buf);
      
      chunks::get(c).setUpdate(u);
    }
  }
  
  void previous(void) {
    printf("WARNING: Cannot go back to previous timesteps for an online visualization\n");
  }
  
};

/*********** File Mesh Source *************/
class fileMeshDataSource : public meshDataSource {
  int nTimesteps;
  int curStep;

  char dirRoot[1024];

  FILE *openFile(int timestep, int forChunk) {
    char fName[1000];
    sprintf(fName,"%s/%d/%d.dat",dirRoot,timestep,forChunk);
    return fopen(fName,"rb");
  }
  int fileLength(FILE *f) {
  	fseek(f,0,SEEK_END);
  	long len=ftell(f);
  	fseek(f,0,SEEK_SET);
	return len;
  }
  
 void countChunks(void){
   chunks::n=0;
   FILE *f;
   while (NULL!=(f=openFile(curStep,chunks::n))) {
     chunks::n++; fclose(f);
   }
 }
  
  void countTimesteps(void){
    nTimesteps=0;
    FILE *f;
    while (NULL!=(f=openFile(nTimesteps,0))) {
      nTimesteps++; fclose(f);
    }
  }
  
public:
  fileMeshDataSource(const char *dirName_) {
    if(dirName_){
      char *dirBase=(char*)&dirName_[strlen(dirName_)-1];
      if (*dirBase=='/') dirBase--;                 // back over a trailing '/'
      while (dirBase!=dirName_ && isdigit(*dirBase)) // backup to beginning of trailing number
        dirBase--;
      dirBase++;
      curStep=atoi(dirBase);
      *dirBase = 0; // null terminate, cutting off the digits at the end
      strcpy(dirRoot,dirName_);
      
      printf("dirRoot=%s\n", dirRoot);
      
      countChunks();
      countTimesteps();

    }
    else {
	  printf("fileMeshDataSource called with no dirname\n");
    }
  }
  
  
  
  int getNumPe(void) {
    //FIXME: may have more chunks than PEs
    return chunks::n;
  }
  int getNumChunks(void) {
    return chunks::n;
  }
  
  
  void update(void){
    if(curStep < nTimesteps-1) {     
      curStep++;
      countChunks();
      // Open the current directory's files:
      for (int c=0;c<getNumChunks();c++) {
        FILE *f=openFile(curStep,c);
        if (f==NULL) bad("Can't open NetFEM input file");
        long size=fileLength(f);
        void *buf=malloc(size);
        if (size!=fread(buf,1,size,f)) bad("Can't read NetFEM input file");
        
        NetFEM_update *u=new NetFEM_update(0,0,0);
        unpack(*u,buf,size);
        free(buf);
        chunks::get(c).setUpdate(u);
        fclose(f);  
        
      }
    }
    updateNextPrevButtons();
  }
  
  void previous(void) {
    if(curStep>0) {
      curStep--;
      countChunks();
      // Open the current directory's files:
      for (int c=0;c<getNumChunks();c++) {
        FILE *f=openFile(curStep,c);
        if (f==NULL) bad("Can't open NetFEM input file");
        long size=fileLength(f);
        void *buf=malloc(size);
        if(size <= 0){
          printf("ERROR: size of some file for timestep %d is not greater than zero\n", curStep);
          exit(2);
        }
        
        if (size!=fread(buf,1,size,f)) bad("Can't read NetFEM input file");
        
        NetFEM_update *u=new NetFEM_update(0,0,0);
        unpack(*u,buf,size);
        free(buf);
        chunks::get(c).setUpdate(u);
        fclose(f);
      }
    }	
    updateNextPrevButtons();
  }

  void updateNextPrevButtons(){
    printf("curStep=%d nTimesteps=%d\n", curStep, nTimesteps);
  
    if(curStep >= nTimesteps-1){
      // disable the next button, enable the previous button
      chk(Tcl_Eval(interp,".main.f.ts.b1 configure -state disabled"));
      chk(Tcl_Eval(interp,".main.f.ts.b0 configure -state active"));
    }
    else if(curStep <= 0){
      // enable the next button, disable the previous
      chk(Tcl_Eval(interp,".main.f.ts.b1 configure -state normal"));
      chk(Tcl_Eval(interp,".main.f.ts.b0 configure -state disabled"));
    }
    else{
      // enable both buttons
      chk(Tcl_Eval(interp,".main.f.ts.b1 configure -state normal"));
      chk(Tcl_Eval(interp,".main.f.ts.b0 configure -state normal"));
    }


  }
  
  
};


/************* Vtk/Gui interface ***************/
static outputFilter *patchworkOF=new nodeOutputFilter(new patchworkFieldFilter);
static geomFilter *cloudGF=new cloudGeomFilter;
static geomFilter *triGF=NULL;
static dispFilter *noDF=new noDispFilter;

static vtkRenderWindow *renWin=NULL;
static vtkScalarBarActor *scl=NULL;
static vtkActor **chunkActors=NULL;
static vtkActor **tubeActors=NULL;
static vtkActor **ballsActors=NULL;
static vtkActor2D **pointLabelActors=NULL;
static vtkActor2D **cellLabelActors=NULL;
static vtkRenderer *ren1;
static vtkRenderWindowInteractor *iren;

static int *chunks_enabled;
static int tubes_enabled;
static int cell_labels_enabled;
static int point_labels_enabled;
static int colors_inverted;

static int nButtons;     // These determine unique identifiers for each button or checkbox
static int nCheckboxes;

const char *guiRerender(void) 
{ //Update the displayed geometry
	int c;
	outputFilter *o=outputFilter::get();
	o->beginRendering();
	for (c=0;c<chunks::n;c++)
		chunks::get(c).addGeometry(o);
	for (c=0;c<chunks::n;c++)
		chunks::get(c).updateRange(o);
	renWin->Render();
	return NULL;
}

void guiInit(void)
{
  printf("guiInit()\n");

  nButtons=0;
  nCheckboxes=0;
  
  tubes_enabled=0;
  point_labels_enabled=0;
  cell_labels_enabled=0;

  colors_inverted = 0;

	ren1=vtkRenderer::New();
	renWin=vtkRenderWindow::New();
  renWin->AddRenderer(ren1);
  renWin->SetSize(600,600);
  
  vtkScalarsToColors *lut=NULL;
  chunkActors = new vtkActor*[chunks::n];
  pointLabelActors = new vtkActor2D*[chunks::n];
  cellLabelActors = new vtkActor2D*[chunks::n];
  tubeActors = new vtkActor*[chunks::n];
  ballsActors = new vtkActor*[chunks::n];
  chunks_enabled = new int[chunks::n];

  for (int c=0;c<chunks::n;c++) {

		vtkPolyDataMapper *map=chunks::get(c).getMap();
		lut=map->GetLookupTable();
		chunkActors[c]=vtkActor::New();
		chunkActors[c]->SetMapper(map);
    ren1->AddActor(chunkActors[c]);
    chunks_enabled[c] = 1;

		vtkLabeledDataMapper *pointLabelMap=chunks::get(c).getPointLabelMap();
		pointLabelActors[c]=vtkActor2D::New();
		pointLabelActors[c]->SetMapper(pointLabelMap);
    ((vtkLabeledDataMapper*)pointLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 1, 1, 1 );

		vtkLabeledDataMapper *cellLabelMap=chunks::get(c).getCellLabelMap();
		cellLabelActors[c]=vtkActor2D::New();
		cellLabelActors[c]->SetMapper(cellLabelMap);
    ((vtkLabeledDataMapper*)cellLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 1, 1, 1 );

    vtkPolyDataMapper *tubeMap=chunks::get(c).getTubesMap();
		tubeActors[c]=vtkActor::New();
		tubeActors[c]->SetMapper(tubeMap);
    //ren1->AddActor(tubeActors[c]);

    vtkPolyDataMapper *ballMap=chunks::get(c).getBallsMap();
		ballsActors[c]=vtkActor::New();
		ballsActors[c]->SetMapper(ballMap);
    ballsActors[c]->GetProperty()->SetColor(0.9,0.0,0.0);
    //   ren1->AddActor(ballsActors[c]);
	}
  
  // When using X and Tcl, we need this special interactor
#if USE_X==1
  iren=vtkXRenderWindowTclInteractor::New();
#else
  iren=vtkRenderWindowInteractor::New();
#endif
  iren->SetRenderWindow(renWin);
  iren->Initialize();
	
  scl=vtkScalarBarActor::New();
  scl->SetLookupTable(lut);
  scl->SetHeight(0.5);
  scl->SetWidth(0.15);
  scl->SetPosition(0,0.2);
  scl->SetTitle("Chunk Number");
  ren1->AddActor2D(scl);

  // Set initial colors
  ren1->SetBackground(0,0,0);
  scl->GetTitleTextProperty()->SetColor(1,1,1);
  scl->GetLabelTextProperty()->SetColor(1,1,1);
  
  guiRerender();
}
void writeImageNamedJPEG(const char *filename) 
{
	vtkWindowToImageFilter *iw = vtkWindowToImageFilter::New();
	iw->SetInput(renWin);
  iw->SetMagnification(4);                 // TODO: Make this parameter configurable
	vtkJPEGWriter *fi = vtkJPEGWriter::New();  // TODO: Give Option of vtkPostScriptWriter, and vtkJPEGWriter
	fi->SetInput(iw->GetOutput());
	fi->SetFileName(filename);
	printf("Saving image data to '%s'\n",filename);
  fi->Write();
	fi->Delete();
	iw->Delete();
  guiRerender();
}
void writeImageNamedPS(const char *filename) 
{
	vtkWindowToImageFilter *iw = vtkWindowToImageFilter::New();
	iw->SetInput(renWin);
  iw->SetMagnification(4);                 // TODO: Make this parameter configurable
	vtkPostScriptWriter *fi = vtkPostScriptWriter::New();  // TODO: Give Option of vtkPostScriptWriter, and vtkJPEGWriter
	fi->SetInput(iw->GetOutput());
	fi->SetFileName(filename);
	printf("Saving image data to '%s'\n",filename);
  fi->Write();
	fi->Delete();
	iw->Delete();
  guiRerender();
}
void writeImageNamedPNG(const char *filename) 
{
	vtkWindowToImageFilter *iw = vtkWindowToImageFilter::New();
	iw->SetInput(renWin);
  iw->SetMagnification(4);                 // TODO: Make this parameter configurable
	vtkPNGWriter *fi = vtkPNGWriter::New();  // TODO: Give Option of vtkPostScriptWriter, and vtkJPEGWriter
	fi->SetInput(iw->GetOutput());
	fi->SetFileName(filename);
	printf("Saving image data to '%s'\n",filename);
  fi->Write();
  fi->Delete();
	iw->Delete();
  guiRerender();
}


const char *writeImageJPEG(void)
{
	char fName[100];
	sprintf(fName,"netfem_%06d.jpg",chunks::get(0).getUpdate()->getTimestep());
	writeImageNamedJPEG(fName);
	return NULL;
}
const char *writeImagePS(void)
{
	char fName[100];
	sprintf(fName,"netfem_%06d.ps",chunks::get(0).getUpdate()->getTimestep());
	writeImageNamedPS(fName);
	return NULL;
}
const char *writeImagePNG(void)
{
	char fName[100];
	sprintf(fName,"netfem_%06d.png",chunks::get(0).getUpdate()->getTimestep());
	writeImageNamedPNG(fName);
	return NULL;
}


static const char *updateMesh(void);
static const char *previousMesh(void);
const char *renderMoviePNG(void)
{
	while (1) {
		writeImagePNG();
		if (updateMesh()!=NULL)
			break;
		guiRerender();
	} 
	return NULL;
}

class filterBundle {
	outputFilter *of;
	geomFilter *gf;
	char title[200];
public:
	filterBundle(outputFilter *of_,geomFilter *gf_,const char *title_) 
		:of(of_), gf(gf_) {strcpy(title,title_);}
	void set(void) {
		scl->SetTitle(title);
		outputFilter::set(of);
		if (gf->getType()!=geomFilter::get()->getType())
			geomFilter::set(gf);
	}
};

const char *guiSetOF(void *ptr) {
	outputFilter::set((outputFilter *)ptr);
	return guiRerender();
}
const char *guiSetGF(void *ptr) {
	geomFilter::set((geomFilter *)ptr);
	return guiRerender();
}

const char *guiSetEF(void *ptr)
{
	((filterBundle *)ptr)->set();
	return guiRerender();
}

const char *guiSetDF(void *ptr) {
	dispFilter::set((dispFilter *)ptr);
	return guiRerender();
}


void updateAllActors(void){
  // Update actors so that the applicable ones are shown. 
  // We only want desired chunks and special features 
  // such as labels only if selected.
  for(int c=0;c<chunks::n;c++){

    if(chunks_enabled[c]){
      ren1->AddActor(chunkActors[c]);
 
      if(point_labels_enabled)
        ren1->AddActor(pointLabelActors[c]);
      else
        ren1->RemoveActor(pointLabelActors[c]);
      
      if(cell_labels_enabled)
        ren1->AddActor(cellLabelActors[c]);
      else
        ren1->RemoveActor(cellLabelActors[c]);

    
      if(tubes_enabled){
        ren1->AddActor(tubeActors[c]);
        ren1->AddActor(ballsActors[c]);
      }
      else{
        ren1->RemoveActor(tubeActors[c]);
        ren1->RemoveActor(ballsActors[c]);
      }
      
    }
    else{
      ren1->RemoveActor(chunkActors[c]);
      ren1->RemoveActor(pointLabelActors[c]);
      ren1->RemoveActor(cellLabelActors[c]);
      ren1->RemoveActor(tubeActors[c]);
      ren1->RemoveActor(ballsActors[c]);
    }
    
  }
}


// Add or Remove a chunk when its checkbox is modified
const char *guiSetChunks(void *ptr) {
  int whichChunk = ((int*)ptr)[0];
  if(chunks_enabled[whichChunk]){
    printf("removing chunk %d\n", whichChunk);
    chunks_enabled[whichChunk]=0;
  }
  else{
    printf("adding chunk %d\n", whichChunk);
    chunks_enabled[whichChunk]=1;
  }
  
  updateAllActors();
	return guiRerender();
}

const char *guiSetScalar(void *ptr) {
  int *param = (int*)ptr;
    
  if(param[0]){ // a state variable which we associate with the status of the checkbox.
    param[0] = 0;
    printf("removing scalar bar\n");
    ren1->RemoveActor(scl);
  }
  else{
    param[0] = 1;
    printf("adding scalar bar\n");
    ren1->AddActor(scl);
  }
	return guiRerender();
}


const char *guiSetTube(void *ptr) {
  if(tubes_enabled)
    tubes_enabled=0;
  else
    tubes_enabled=1;
  
  updateAllActors();
	return guiRerender();
}

const char *guiSetPointLabels(void *ptr) {
  if(point_labels_enabled)
    point_labels_enabled=0;
  else
    point_labels_enabled=1;
  updateAllActors();
	return guiRerender();
}

const char *guiSetCellLabels(void *ptr) {
  if(cell_labels_enabled)
    cell_labels_enabled=0;
  else
    cell_labels_enabled=1;
  updateAllActors();
	return guiRerender();
}

static const char *guiSetCamera(void) {
  ren1->GetActiveCamera()->SetFocalPoint(0,0,0);
  ren1->GetActiveCamera()->SetPosition(0,0,1);
  ren1->GetActiveCamera()->ComputeViewPlaneNormal();
  ren1->GetActiveCamera()->SetViewUp(0,1,0);
  ren1->GetActiveCamera()->OrthogonalizeViewUp();
  ren1->ResetCamera();
	return guiRerender();
}

const char *guiSetColors(void *ptr) {
  printf("inverting colors\n");

  if(colors_inverted){ // We will set the colors to their inital uninverted state
    colors_inverted=0;
    ren1->SetBackground(0,0,0);

    scl->GetTitleTextProperty()->SetColor(1,1,1);
    scl->GetLabelTextProperty()->SetColor(1,1,1);
    for(int c=0;c<chunks::n;c++){
      ((vtkLabeledDataMapper*)pointLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 1, 1, 1 );
      ((vtkLabeledDataMapper*)cellLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 1, 1, 1 );
    }
    
  }
  else{ // We will set the colors to an inverted state
    colors_inverted=1;
    ren1->SetBackground(1,1,1);
    
    scl->GetTitleTextProperty()->SetColor(0,0,0);
    scl->GetLabelTextProperty()->SetColor(0,0,0);
    for(int c=0;c<chunks::n;c++){
      ((vtkLabeledDataMapper*)pointLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 0, 0, 0 );
      ((vtkLabeledDataMapper*)cellLabelActors[c]->GetMapper())->GetLabelTextProperty()->SetColor( 0, 0, 0 );
    }
  }

  updateAllActors();
	return guiRerender();
}

/************* Main window widgets ***************/
static const char *createMesh(const char *str) {
	if (strchr(str,':')!=NULL)
		/* Has a colon: treat as a CCS location */
		meshSource=new ccsMeshDataSource(str);
	else	/* No colon: treat as a directory name */
		meshSource=new fileMeshDataSource(str);
	return NULL;
}

static const char *updateMesh(void) {
	char str[200];
	chk(Tcl_Eval(interp,"setStatus {Requesting current step...}"));
	meshSource->update();
	
	const NetFEM_update *last_update;
	int minTS=1000000000,maxTS=-1000000000;
  printf("chunks::n=%d\n", chunks::n);
	for (int c=0;c<chunks::n;c++) {
		const NetFEM_update *u=chunks::get(c).getUpdate();
		if (u->getTimestep()<minTS) minTS=u->getTimestep();
		if (u->getTimestep()>maxTS) maxTS=u->getTimestep(); 
		last_update=u;
	}
	
	status("Updating timestep display");
	if (minTS==maxTS)
	  sprintf(str,"Timestep %d (%dD data)",minTS,last_update->getDim());
	else
	  sprintf(str,"Timesteps %d-%d (%dD data)",minTS,maxTS,last_update->getDim());
	chk(Tcl_VarEval(interp,"  .main.f.ts.l configure -text {",str,"}",NULL));
	chk(Tcl_Eval(interp,"setStatus {Ready}"));
	return NULL;
}

static const char *previousMesh(void) {
	char str[200];
	chk(Tcl_Eval(interp,"setStatus {Requesting current step...}"));
	meshSource->previous();
	
	const NetFEM_update *last_update;
	int minTS=1000000000,maxTS=-1000000000;
  printf("chunks::n=%d\n", chunks::n);
	for (int c=0;c<chunks::n;c++) {
		const NetFEM_update *u=chunks::get(c).getUpdate();
		if (u->getTimestep()<minTS) minTS=u->getTimestep();
		if (u->getTimestep()>maxTS) maxTS=u->getTimestep(); 
		last_update=u;
	}
	
	status("Updating timestep display");
	if (minTS==maxTS)
	  sprintf(str,"Timestep %d (%dD data)",minTS,last_update->getDim());
	else
	  sprintf(str,"Timesteps %d-%d (%dD data)",minTS,maxTS,last_update->getDim());
	chk(Tcl_VarEval(interp,"  .main.f.ts.l configure -text {",str,"}",NULL));
	chk(Tcl_Eval(interp,"setStatus {Ready}"));
	return NULL;
}

typedef enum {sLeft=0,sTop,sRight,sBottom} lSide;
static const char *lSide2str[]={"left","top","right","bottom"};
static void addLabel(const char *w,const char *path,const char *title,lSide side=sTop) {
	chk(Tcl_VarEval(interp,
	"  label ",w,".",path," -text {",title,"}\n",
	"  pack ",w,".",path," -side ",lSide2str[side]," -anchor nw\n",
	NULL));
}
static void addSpacer(const char *w) {
	addLabel(w,"spc"," ",sLeft);
}
static void addFrame(const char *w) {
	chk(Tcl_VarEval(interp,
	"  frame ",w,"\n",
	"  pack ",w," -side top -anchor nw -fill x -expand yes\n",
	NULL));
}

class bar {
	char w[200];
public:
	bar(const char *gw,const char *path,int indent,const char *name)
	{
		if (path==0 || strlen(path)==0)
			sprintf(w,"%s",gw);
		else
			sprintf(w,"%s.%s",gw,path);
		addFrame(w);
		for (int i=0;i<indent;i++) addSpacer(w);
		addLabel(w,"l",name,sLeft);
	}
	void button(const char *butName,const char *cmd,const void *cmdParam,lSide s=sLeft)
	{
		char str[200],bp[200];
		sprintf(str,"%s %p",cmd,cmdParam);
		sprintf(bp,"%s.b%d",w,nButtons++);
    printf("Creating a button with path= %s\n", bp);
		chk(Tcl_VarEval(interp,
		"c2tcl_eval {\n",
		"  button ",bp," -text {",butName,"} -command {c2tcl_eval {",str,"}}\n",
		"  pack ",bp," -side ",lSide2str[s],"\n",
		"}\n",
		NULL));	
	}
	void checkbox(const char *cmd,const void *cmdParam,lSide s=sLeft, int selected=1)
	{
		char str[200],bp[200];
		sprintf(str,"%s %p",cmd,cmdParam);
		sprintf(bp,"%s.check%d",w,nCheckboxes++);
    printf("bp=%s\n", bp);

    if(selected) {
      chk(Tcl_VarEval(interp,
                      "c2tcl_eval {\n",
                      "  checkbutton ",bp," -command {c2tcl_eval {",str,"}}\n",
                      "  pack ",bp," -side ",lSide2str[s],"\n",
                      "  ",bp," select \n",
                      "}\n",
                      NULL));
    }
    else {
      chk(Tcl_VarEval(interp,
                      "c2tcl_eval {\n",
                      "  checkbutton ",bp," -command {c2tcl_eval {",str,"}}\n",
                      "  pack ",bp," -side ",lSide2str[s],"\n",
                      "}\n",
                      NULL));
    }
      

	}


};

void buttonBarWidget(
	//Widget location
	const char *gw,const char *path,
	//Widget indent and label
	int indent,const char *name,
	//Button (name and command to execute when pressed)
	const char *butName=NULL,const char *cmd=NULL,const void *cmdParam=NULL)
{
	bar b(gw,path,indent,name);
	if (butName!=0) b.button(butName,cmd,cmdParam);
}

static const char *axisNames[]={"X","Y","Z"};
static const char *numNames[]={"1","2","3","4","5","6","7","8","9","10"};

//Write a human-readable version of the name of this field into str. 
// Return a table of the names of its components
const char **fieldName(const NetFEM_doubleField *f,char *str)
{
	const char **subNames=numNames;
	if (f->getSpatial()) {
		sprintf(str,"Vector ");
		subNames=axisNames;
	} else if (f->getStride()>1)
		sprintf(str,"%d-Scalar ",f->getStride());
	else
		sprintf(str,"");
	sprintf(&str[strlen(str)],f->getName());
	return subNames;
}

void nodeFieldWidgets(const char *w,const NetFEM_doubleField *f,geomFilter *gf,int fNo) {
	char str[200];
	const char **subNames=fieldName(f,str);
	bar b(w,"",1,str);
	if (f->getStride()==1)
		b.button("Show","guiSetEF",new filterBundle(new nodeOutputFilter(
			new scalarFieldFilter(new nodeFieldLoc(fNo,0))),gf,f->getName()),sRight);
	else {
		if (f->getSpatial()) {
			b.button("Disp","guiSetDF",new nodeDispFilter(fNo),sLeft);
			sprintf(str,"Mag %s",f->getName());
			b.button("Mag","guiSetEF",new filterBundle(new nodeOutputFilter(
				new magFieldFilter(new nodeFieldLoc(fNo,0))),gf,str),sRight);
		}
		for (int i=f->getStride()-1;i>=0;i--) {
			sprintf(str,"%s %s",subNames[i],f->getName());
			b.button(subNames[i],"guiSetEF",new filterBundle(new nodeOutputFilter(
				new scalarFieldFilter(new nodeFieldLoc(fNo,i))),gf,str),sRight);
		}
	}
}
void nodeWidgets(const char *w,const NetFEM_nodes *n) {
	char str[200],lw[200];
	sprintf(str,"%d Nodes:",n->getItems());
	buttonBarWidget(w,"spacer",0," ");
	buttonBarWidget(w,"n",0,str, "Point Nodes","guiSetGF",cloudGF);
	//Skip first field (it's just the coordinates)
	int i;
	for (i=1;i<n->getFields();i++)
	{
		sprintf(lw,"%s.f%d",w,i);
		nodeFieldWidgets(lw,&n->getField(i),triGF,i);
	}
}
void elemFieldWidgets(const char *w,const NetFEM_doubleField *f,geomFilter *gf,int eType,int fNo) {
	char str[200];
	const char **subNames=fieldName(f,str);
	bar b(w,"",1,str);
	if (f->getStride()==1)
		b.button("Show","guiSetEF",new filterBundle(new elemOutputFilter(
			new scalarFieldFilter(new elemFieldLoc(eType,fNo,0)),eType),gf,f->getName()),sRight);
	else {
		if (f->getSpatial()) {
			sprintf(str,"Mag %s",f->getName());
			b.button("Mag","guiSetEF",new filterBundle(new elemOutputFilter(
				new magFieldFilter(new elemFieldLoc(eType,fNo,0)),eType),gf,str),sRight);
		}
		for (int i=f->getStride()-1;i>=0;i--) {
			sprintf(str,"%s %s",subNames[i],f->getName());
			b.button(subNames[i],"guiSetEF",new filterBundle(new elemOutputFilter(
				new scalarFieldFilter(new elemFieldLoc(eType,fNo,i)),eType),gf,str),sRight);
		}
	}
}

void elemWidgets(const char *w,const NetFEM_elems *e,int eType) {
	char str[200],lw[200];
	sprintf(str,"%d %s:",e->getItems(),e->getName());
	buttonBarWidget(w,"spacer",0," ");
	bar b(w,"n",0,str);
	if (e->getNodesPer()==3) 
		b.button("Fill","guiSetGF",new triGeomFilter(eType));
	  b.button("Wireframe","guiSetGF",new outlineGeomFilter(eType));
	int i;
	geomFilter *gf=new elemGeomFilter(eType);
	for (i=0;i<e->getFields();i++)
	{
		sprintf(lw,"%s.f%d",w,i);
		elemFieldWidgets(lw,&e->getField(i),gf,eType,i);
	}
}

void chunkWidgets(char *w, int numChunks) {
  bar b(w,"chunks",0,"Enable Chunks");
  for(int i=0;i<numChunks && i<MAX_CHUNK_CHECKBOXES;i++){
    int *param = new int[2];
    param[0] = i;
    param[1] = 1;
    lSide side=sLeft;
    b.checkbox("guiSetChunks",param,side,1);
  }
}

void EnableTubesWidgets(char *w) {
  bar b(w,"tubes",0,"Enable TubeView");
  int *param = new int;
   param[0] = 1;
   lSide side=sLeft;
   b.checkbox("guiSetTube",param,side, 0);
}

void EnableLabelsWidgets(char *w) {

  {
    bar b(w,"pointLabels",0,"Point Labels");
    int *param = new int;
    param[0] = 1;
    lSide side=sLeft;
    b.checkbox("guiSetPointLabels",param,side,0);
  }

  {
    bar b(w,"cellLabels",0,"Cell Labels");
    int *param = new int;
    param[0] = 1;
    lSide side=sLeft;
    b.checkbox("guiSetCellLabels",param,side,0);
  }

}

void ScalarBarWidgets(char *w) {
  bar b(w,"scalarbar",0,"Enable Scalar Bar");
  int *param = new int;
   param[0] = 1;
   lSide side=sLeft;
   b.checkbox("guiSetScalar",param,side,1);
}

void CameraBarWidgets(char *w) {
  bar b(w,"camerabar",0,"Camera Bar");
  lSide side=sLeft;
   b.button("Reset Camera","guiSetCamera",NULL);
}

void ColorWidgets(char *w) {
  bar b(w,"colorbar",0,"Color Bar");
  lSide side=sLeft;
  b.button("Invert Colors","guiSetColors",NULL);
}


/************ Startup Routine ************/
const char *ccsWidgets(char *w) {
	char str[200];
	
	outputFilter::set(patchworkOF);

//Get cursory information about the computation:
	int nPe=meshSource->getNumPe();
	sprintf(str,"Running on %d processors",nPe);
	addLabel(w,"npe",str);
	
	int nChunks=meshSource->getNumChunks();
	sprintf(str,"global nChunks\n  set nChunks %d\n",nChunks);
	tcl_eval(str);
	chunks::init(nChunks);

//Begin to create the GUI		
  bar timestepb(w,"ts",0,"-Placeholder");
  timestepb.button("Previous","v_step_back",NULL);
  timestepb.button("Next","v_step",NULL);

//Load each chunk
	updateMesh();
	const NetFEM_update *u=chunks::get(0).getUpdate();
	
//Figure out which element types are displayable and set the initial filters
	int ne;
	triGF=NULL;
	for (ne=0;ne<u->getElems();ne++)
		if (u->getElem(ne).getNodesPer()==3)
		{
			triGF=new triGeomFilter(ne);
		}
	if (triGF==NULL)
		triGF=cloudGF;
	geomFilter::set(triGF);
	dispFilter::set(noDF);
	
	sprintf(str,"%d Chunks",nChunks);	
	{
    bar b(w,"chk",0,str);
    b.button("Show","guiSetEF",new filterBundle(patchworkOF,triGF,"Chunk Number"),sRight);
  }

//Set up the GUI for chunks
  chunkWidgets(w, nChunks);
  

//Set up the GUI for the nodes
	char lw[200];
	sprintf(lw,"%s.n",w);
	addFrame(lw);
	nodeWidgets(lw,&u->getNodes());

//Set up the GUI for the elements
	for (ne=0;ne<u->getElems();ne++) {
		sprintf(lw,"%s.e%d",w,ne);
		addFrame(lw);
		elemWidgets(lw,&u->getElem(ne),ne);
	}

  EnableLabelsWidgets(w);
  EnableTubesWidgets(w);
  ScalarBarWidgets(w);
  CameraBarWidgets(w);
  ColorWidgets(w);

//Start VTK
	guiInit();

	status("Done with ccsWidgets");
	
	return NULL;
}

/************* Base Tcl calls ************/
void tcl_eval(const char *string) {
	chk(Tcl_Eval(interp,(char *)string));
}

static int str2tcl(const char *err) {
	if (err==NULL) return TCL_OK;
	Tcl_SetResult(interp, (char *)err, TCL_STATIC);
	return TCL_ERROR;
}

typedef const char *(*stringFn)(char *arg);
static int callStringFn(ClientData cd,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[])
{
	stringFn f=(stringFn)cd;
	if (objc!=2) return str2tcl("this routine takes exactly one string as argument");
	else {
		char *s=Tcl_GetStringFromObj(objv[1],NULL);
		return str2tcl( (f)(s) );
	}
}

typedef const char *(*voidPtrFn)(void *);
static int callPtrFn(ClientData cd,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[])
{
	voidPtrFn f=(voidPtrFn)cd;
	if (objc!=2) return str2tcl("this routine takes a pointer as an argument");
	else {
		char *s=Tcl_GetStringFromObj(objv[1],NULL);
		void *p;
		if (1!=sscanf(s,"%p",&p)) return str2tcl("Cannot parse argument as a pointer");
		return str2tcl( (f)(p) );
	}
}

typedef const char *(*voidFn)(void);
static int callVoidFn(ClientData cd,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[])
{
	voidFn f=(voidFn)cd;
	if (objc!=1) return str2tcl("this routine takes no arguments");
	else {
		return str2tcl( (f)() );
	}
}

static int modalCount=0;
static int Cmd_modalBegin(ClientData cd,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[])
{
	int modalStart=modalCount;
	modalCount++;
	while (modalCount!=modalStart)
		Tcl_DoOneEvent(0);
	return TCL_OK;
}
static int Cmd_modalEnd(ClientData cd,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[])
{
	modalCount--;
	return TCL_OK;
}


/*Register commands before reading TCL routines*/
void tkMyAppInit(Tcl_Interp *interp)
{
	//Application:
	Tcl_CreateObjCommand(interp,"createMesh",callStringFn,(ClientData)createMesh,NULL);
	Tcl_CreateObjCommand(interp,"updateMesh",callVoidFn,(ClientData)updateMesh,NULL);
	Tcl_CreateObjCommand(interp,"previousMesh",callVoidFn,(ClientData)previousMesh,NULL);
	Tcl_CreateObjCommand(interp,"ccsWidgets",callStringFn,(ClientData)ccsWidgets,NULL);
	Tcl_CreateObjCommand(interp,"guiRerender",callVoidFn,(ClientData)guiRerender,NULL);
	Tcl_CreateObjCommand(interp,"writeImagePNG",callVoidFn,(ClientData)writeImagePNG,NULL);
	Tcl_CreateObjCommand(interp,"writeImageJPEG",callVoidFn,(ClientData)writeImageJPEG,NULL);
	Tcl_CreateObjCommand(interp,"writeImagePS",callVoidFn,(ClientData)writeImagePS,NULL);
	Tcl_CreateObjCommand(interp,"renderMoviePNG",callVoidFn,(ClientData)renderMoviePNG,NULL);
	Tcl_CreateObjCommand(interp,"guiSetOF",callPtrFn,(ClientData)guiSetOF,NULL);
	Tcl_CreateObjCommand(interp,"guiSetGF",callPtrFn,(ClientData)guiSetGF,NULL);
	Tcl_CreateObjCommand(interp,"guiSetEF",callPtrFn,(ClientData)guiSetEF,NULL);
	Tcl_CreateObjCommand(interp,"guiSetDF",callPtrFn,(ClientData)guiSetDF,NULL);
	
	Tcl_CreateObjCommand(interp,"guiSetChunks",callPtrFn,(ClientData)guiSetChunks,NULL);
  Tcl_CreateObjCommand(interp,"guiSetScalar",callPtrFn,(ClientData)guiSetScalar,NULL);
  Tcl_CreateObjCommand(interp,"guiSetTube",callPtrFn,(ClientData)guiSetTube,NULL);
  Tcl_CreateObjCommand(interp,"guiSetPointLabels",callPtrFn,(ClientData)guiSetPointLabels,NULL);
  Tcl_CreateObjCommand(interp,"guiSetCellLabels",callPtrFn,(ClientData)guiSetCellLabels,NULL);
  Tcl_CreateObjCommand(interp,"guiSetCamera",callPtrFn,(ClientData)guiSetCamera,NULL);
  Tcl_CreateObjCommand(interp,"guiSetColors",callPtrFn,(ClientData)guiSetColors,NULL);


	//Utility:
	Tcl_CreateObjCommand(interp,"modalBegin",Cmd_modalBegin,0,NULL);
	Tcl_CreateObjCommand(interp,"modalEnd",Cmd_modalEnd,0,NULL);
}

/*Start the application proper*/
void tkMyAppStart(Tcl_Interp *interp,char **argv)
{
	skt_set_abort(tcl_skt_abortFn);
	skt_set_idle(tcl_skt_idleFn);
	
	argv++; //skip over app. name
	if (*argv!=NULL) {//user passed parameter on command line
		if (Tcl_VarEval(interp,"c2tcl_eval {startAppWith {",
			*argv,"} }",NULL)==TCL_ERROR) bad("error during startup");
	} else {
		if (Tcl_Eval(interp,"c2tcl_eval {startApp}")==TCL_ERROR) bad("error during arg startup");
	}
	printf("App started\n");
}

