/*****************************************************************************
 * A few useful built-in CPD and CCS handlers.
 *****************************************************************************/

#include "converse.h"

#include <errno.h>
#include <string.h>
#include <sys/stat.h>		// for chmod

#include "ckhashtable.h"
#include "conv-ccs.h"
#include "debug-charm.h"
#include "sockRoutines.h"
#include "charm.h"
#include "middle.h"
#include "cklists.h"
#include "register.h"
//#include "queueing.h"
#include <unistd.h>

#if CMK_CCS_AVAILABLE && !defined(_WIN32)

#include "ck.h"

CkVec<DebugEntryInfo> _debugEntryTable;
CpdPersistentChecker persistentCheckerUselessClass;

void CpdFinishInitialization() {
#ifndef CMK_OPTIMIZE
  _debugEntryTable.reserve(_entryTable.size());
#endif
}

extern "C" void resetAllCRC();
extern "C" void checkAllCRC(int report);

typedef struct DebugRecursiveEntry {
  int previousChareID;
  int alreadyUserCode;
  char *memoryBackup;
} DebugRecursiveEntry;

CkQ<DebugRecursiveEntry> _debugData;

// Function called right before an entry method
void CpdBeforeEp(int ep, void *obj) {
#ifndef CMK_OPTIMIZE
  if (CpvAccess(cmiArgDebugFlag)) {
    DebugRecursiveEntry entry;
    entry.previousChareID = setMemoryChareIDFromPtr(obj);
    entry.alreadyUserCode = _entryTable[ep]->inCharm ? 0 : 1;
    entry.memoryBackup = NULL;
    _debugData.push(entry);
    setMemoryStatus(entry.alreadyUserCode);
    //if (_debugEntryTable[ep].isBreakpoint) printf("CpdBeforeEp breakpointed %d\n",ep);
    memoryBackup = &_debugData.peek().memoryBackup;
    if (!_entryTable[ep]->inCharm) {
      CpdResetMemory();
    }
  }
#endif
}

// Function called right after an entry method
void CpdAfterEp(int ep) {
#ifndef CMK_OPTIMIZE
  if (CpvAccess(cmiArgDebugFlag)) {
    DebugRecursiveEntry entry = _debugData.peek();
    CkVec<DebugPersistentCheck> &postExecutes = _debugEntryTable[ep].postProcess;
    for (int i=0; i<postExecutes.size(); ++i) {
      postExecutes[i].object->cpdCheck(postExecutes[i].msg);
    }
    memoryBackup = &entry.memoryBackup;
    if (!_entryTable[ep]->inCharm) {
      CpdCheckMemory();
    }
    setMemoryChareID(entry.previousChareID);
    setMemoryStatus(entry.alreadyUserCode);
    _debugData.deq();
  }
#endif
}

/************ Array Element CPD Lists ****************/

/**
  Count array elements going by until they reach this
  range (lo to hi), then start passing them to dest.
*/
template <class T>
class CkArrayElementRangeIterator : public CkLocIterator {
private:
   T *dest;
   CkArray *mgr;
   int cur,lo,hi;
public:
   CkArrayElementRangeIterator(T *dest_,int l,int h)
   	:dest(dest_),mgr(0),cur(0),lo(l),hi(h) {}

  /** Called to iterate only on a specific array manager.
      Returs the number of objects it iterate on.
  */
  int iterate(int start, CkArray *m) {
    cur = start;
    mgr = m;
    mgr->getLocMgr()->iterate(*this);
    cur -= start;
    return cur;
  }

   /** Call add for every in-range array element on this processor */
   void iterate(void)
   { /* Walk the groupTable for arrays (FIXME: get rid of _groupIDTable) */
     int numGroups=CkpvAccess(_groupIDTable)->size();
     for(int i=0;i<numGroups;i++) {
        IrrGroup *obj = CkpvAccess(_groupTable)->find((*CkpvAccess(_groupIDTable))[i]).getObj();
	if (obj->isArrMgr())
	{ /* This is an array manager: examine its array elements */
	  mgr=(CkArray *)obj;
	  mgr->getLocMgr()->iterate(*this);
	}
     }
   }

   // Called by location manager's iterate function
   virtual void addLocation (CkLocation &loc)
   {
     if (cur>=lo && cur<hi)
     { /* This element is in our range-- look it up */
       dest->add(cur,mgr->lookup(loc.getIndex()),mgr->getGroupID().idx);
     }
     cur++;
   }

   // Return the number of total array elements seen so far.
   int getCount(void) {return cur;}
};

/**
  Count charm++ objects going by until they reach this
  range (lo to hi), then start passing them to dest.
*/
template <class T>
class CkObjectRangeIterator {
private:
   T *dest;
   int cur,lo,hi;
public:
   CkObjectRangeIterator(T *dest_,int l,int h)
   	:dest(dest_),cur(0),lo(l),hi(h) {}

   /** Call add for every in-range array element on this processor */
   void iterate(void)
   { /* Walk the groupTable for arrays (FIXME: get rid of _groupIDTable) */
     int numGroups=CkpvAccess(_groupIDTable)->size();
     for(int i=0;i<numGroups;i++) {
       CkGroupID groupID = (*CkpvAccess(_groupIDTable))[i];
        IrrGroup *obj = CkpvAccess(_groupTable)->find(groupID).getObj();
	/*if (obj->isArrMgr())
	{ / * This is an array manager: examine its array elements * /
	  CkArray *mgr=(CkArray *)obj;
          CkArrayElementRangeIterator<T> ait(dest,lo,hi);
	  ait.iterate(cur, mgr);
          cur+=ait.getCount();
	} else {*/
          dest->add(cur,obj,groupID.idx);
          cur++;
        //}
     }
   }

   // Return the number of total array elements seen so far.
   int getCount(void) {return cur;}
};

class ignoreAdd {
public: void add(int cur,Chare *elt,int group) {}
};

/** Examine all the objects on the server returning the name */
class CpdList_objectNames : public CpdListAccessor {
  PUP::er *pp; // Only used while inside pup routine.
  int curGroup;
public:
  virtual const char * getPath(void) const {return "charm/objectNames";}
  virtual size_t getLength(void) const {
    CkObjectRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    CkObjectRangeIterator<CpdList_objectNames> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements
  }
  void add(int cur, Chare *obj, int group) {
    PUP::er &p=*pp;
    beginItem(p,cur);
    p.comment("id");
    char *n = (char*)malloc(30);
    int s=obj->ckDebugChareID(n, 30);
    CkAssert(s > 0);
    p(n,s);
    free(n);
    PUPn(group);
    p.comment("name");
    n=obj->ckDebugChareName();
    p(n,strlen(n));
    free(n);
  }
};

/** Examine a single object identified by the id passed in the request and
    return its type and memory data */
class CpdList_object : public CpdListAccessor {
  PUP::er *pp; //Only used while inside pup routine.
  CpdListItemsRequest *reqq; // Only used while inside pup routine.
public:
  virtual const char * getPath(void) const {return "charm/object";}
  virtual size_t getLength(void) const {
    CkObjectRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    reqq=&req;
    CkObjectRangeIterator<CpdList_object> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements;
  }
  void add(int cur, Chare *obj, int group) {
    PUP::er &p=*pp;
    CpdListItemsRequest &req=*reqq;
    char *n = (char *)malloc(30);
    int s=obj->ckDebugChareID(n, 30);
    CkAssert(s > 0);
    if (req.extraLen == s && memcmp(req.extra, n, s) == 0) {
      // the object match, found!
      beginItem(p,cur);
      int type = obj->ckGetChareType();
      p.comment("type");
      const char *t = _chareTable[type]->name;
      p((char*)t,strlen(t));
      p.comment("value");
      int size = _chareTable[type]->size;
      p((char*)obj,size);
    }
  }
};

/** Coarse: examine array element names */
class CpdList_arrayElementNames : public CpdListAccessor {
  PUP::er *pp; // Only used while inside pup routine.
public:
  virtual const char * getPath(void) const {return "charm/arrayElementNames";}
  virtual size_t getLength(void) const {
    CkArrayElementRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    CkArrayElementRangeIterator<CpdList_arrayElementNames> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements
  }
  void add(int cur,Chare *e,int group)
  { // Just grab the name and nothing else:
    ArrayElement *elt = (ArrayElement*)e;
         PUP::er &p=*pp;
	 beginItem(p,cur);
         p.comment("name");
	 char *n=elt->ckDebugChareName();
	 p(n,strlen(n));
	 free(n);
  }
};

/** Detailed: examine array element data */
class CpdList_arrayElements : public CpdListAccessor {
  PUP::er *pp; // Only used while inside pup routine.
public:
  virtual const char * getPath(void) const {return "charm/arrayElements";}
  virtual size_t getLength(void) const {
    CkArrayElementRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    CkArrayElementRangeIterator<CpdList_arrayElements> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements
  }
  void add(int cur, Chare *e, int group)
  { // Pup the element data
    ArrayElement *elt = (ArrayElement*)e;
    PUP::er &p=*pp;
    beginItem(p,cur);
    //elt->ckDebugPup(p);
    // Now ignore any pupper, just copy all the memory as raw data
    p.comment("name");
    char *n=elt->ckDebugChareName();
    p(n,strlen(n));
    free(n);
    int type = elt->ckGetChareType();
    p.comment("type");
    const char *t = _chareTable[type]->name;
    p((char*)t,strlen(t));
    p.comment("value");
    int size = _chareTable[type]->size;
    p((char*)elt,size);
  }
};

#ifndef __CYGWIN__
#include <rpc/rpc.h>
#endif

size_t hostInfoLength(void *) {return 1;}

void hostInfo(void *itemIter, pup_er pp, CpdListItemsRequest *req) {
  PUP::er &p = *(PUP::er *)pp;
  struct sockaddr_in addr;
  CpdListBeginItem(pp, 0);
#if CMK_HAS_GET_MYADDRESS
  get_myaddress(&addr);
#else
  CmiAbort("hostInfo: get_myaddress does not work on this machine");
#endif
  char *address = (char*)&addr.sin_addr.s_addr;
  PUPv(address, 4);
  int pid = getpid();
  PUPn(pid);
}

/************ Message CPD Lists ****************/
CpvCExtern(void *,debugQueue);
CpvCExtern(int, skipBreakpoint);

// Interpret data in a message in a user-friendly way.
//  Ignores most of the envelope fields used by CkPupMessage,
//  and instead concentrates on user data
void CpdPupMessage(PUP::er &p, void *msg)
{
  envelope *env=UsrToEnv(msg);
  //int wasPacked=env->isPacked();
  int size=env->getTotalsize();
  int prioBits=env->getPriobits();
  int from=env->getSrcPe();
  PUPn(from);
  //PUPn(wasPacked);
  PUPn(prioBits);
  int userSize=size-sizeof(envelope)-sizeof(int)*CkPriobitsToInts(prioBits);
  PUPn(userSize);
  int msgType = env->getMsgIdx();
  PUPn(msgType);
  int msgFor = env->getMsgtype();
  PUPn(msgFor);

  //p.synchronize(PUP::sync_last_system);

  int ep=CkMessageToEpIdx(msg);
  PUPn(ep);

  /* user data */
  p.comment("data");
  p.synchronize(PUP::sync_begin_object);
  if (_entryTable[ep]->messagePup!=NULL)
    _entryTable[ep]->messagePup(p,msg);
  else
    CkMessage::ckDebugPup(p,msg);
  p.synchronize(PUP::sync_end_object);
}

CpvStaticDeclare(void *, lastBreakPointMsg);

//Cpd Lists for local and scheduler queues
class CpdList_localQ : public CpdListAccessor {

public:
  CpdList_localQ() {}
  virtual const char * getPath(void) const {return "converse/localqueue";}
  virtual size_t getLength(void) const {
    int x = CdsFifo_Length((CdsFifo)(CpvAccess(debugQueue)));
    //CmiPrintf("*******Returning fifo length %d*********\n", x);
    //return CdsFifo_Length((CdsFifo)(CpvAccess(CmiLocalQueue)));
    if (CpvAccess(lastBreakPointMsg) != NULL) x++;
    return x;
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    int length = CdsFifo_Length((CdsFifo)(CpvAccess(debugQueue)));
    void ** messages = CdsFifo_Enumerate(CpvAccess(debugQueue));
    int curObj=0;

    if (CpvAccess(lastBreakPointMsg) != NULL) {
      beginItem(p, -1);
      envelope *env=(envelope *)UsrToEnv(CpvAccess(lastBreakPointMsg));
      p.comment("name");
      char *type=(char*)"Breakpoint";
      p(type,strlen(type));
      p.comment("charmMsg");
      p.synchronize(PUP::sync_begin_object);
      CkUnpackMessage(&env);
      CpdPupMessage(p, EnvToUsr(env));
      p.synchronize(PUP::sync_end_object);
    }

    for(curObj=req.lo; curObj<req.hi; curObj++)
      if ((curObj>=0) && (curObj<length))
      {
        beginItem(p,curObj);
        void *msg=messages[curObj]; /* converse message */
        int isCharm=0;
        const char *type="Converse";
        p.comment("name");
        char name[128];
        if (CmiGetHandler(msg)==_charmHandlerIdx) {isCharm=1; type="Local Charm";}
        if (CmiGetXHandler(msg)==_charmHandlerIdx) {isCharm=1; type="Network Charm";}
        sprintf(name,"%s %d: %s (%d)","Message",curObj,type,CmiGetHandler(msg));
        p(name, strlen(name));

        if (isCharm)
        { /* charm message */
          p.comment("charmMsg");
          p.synchronize(PUP::sync_begin_object);
          envelope *env=(envelope *)msg;
          CkUnpackMessage(&env);
          messages[curObj]=env;
          CpdPupMessage(p, EnvToUsr(env));
          //CkPupMessage(p, &messages[curObj], 0);
          p.synchronize(PUP::sync_end_object);
        }
      }
    delete[] messages;

  }
};


/****************** Breakpoints and other debug support **************/

typedef CkHashtableTslow<int,EntryInfo *> CpdBpFuncTable_t;


extern void CpdFreeze(void);
extern void CpdUnFreeze(void);
extern int  CpdIsFrozen(void);

CpvStaticDeclare(int, _debugMsg);
CpvStaticDeclare(int, _debugChare);

CpvStaticDeclare(CpdBpFuncTable_t *, breakPointEntryTable);

//CpvStaticDeclare(void *, lastBreakPointMsg);
CpvStaticDeclare(void *, lastBreakPointObject);
CpvStaticDeclare(int, lastBreakPointIndex);

void CpdBreakPointInit()
{
  CpvInitialize(void *, lastBreakPointMsg);
  CpvInitialize(void *, lastBreakPointObject);
  CpvInitialize(int, lastBreakPointIndex);
  CpvInitialize(int, _debugMsg);
  CpvInitialize(int, _debugChare);
  CpvInitialize(CpdBpFuncTable_t *, breakPointEntryTable);
  CpvAccess(lastBreakPointMsg) = NULL;
  CpvAccess(lastBreakPointObject) = NULL;
  CpvAccess(lastBreakPointIndex) = 0;
  CpvAccess(_debugMsg) = CkRegisterMsg("debug_msg",0,0,0,0);
  CpvAccess(_debugChare) = CkRegisterChare("debug_Chare",0);
  CkRegisterChareInCharm(CpvAccess(_debugChare));
  CpvAccess(breakPointEntryTable) = new CpdBpFuncTable_t(10,0.5,CkHashFunction_int,CkHashCompare_int );
}



static void _call_freeze_on_break_point(void * msg, void * object)
{
      //Save breakpoint entry point index. This is retrieved from msg.
      //So that the appropriate EntryInfo can be later retrieved from the hash table
      //of break point function entries, on continue.

  // If the counter "skipBreakpoint" is not zero we actually do not freeze and deliver the regular message
  if (CpvAccess(skipBreakpoint) > 0) {
    EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CkMessageToEpIdx(msg));
    CkAssert(breakPointEntryInfo != NULL);
    breakPointEntryInfo->call(msg, object);
    CpvAccess(skipBreakpoint) --;
  } else {
      CpvAccess(lastBreakPointMsg) = msg;
      CpvAccess(lastBreakPointObject) = object;
      CpvAccess(lastBreakPointIndex) = CkMessageToEpIdx(msg);
      EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CpvAccess(lastBreakPointIndex));
      CmiPrintf("CPD: Break point reached in proc %d for Function = %s\n",CkMyPe(),breakPointEntryInfo->name);
      CpdFreeze();
  }
}

//ccs handler when pressed the "next" command: deliver only a single message without unfreezing
extern "C"
void CpdDeliverSingleMessage () {
  if (!CpdIsFrozen()) return; /* Do something only if we are in freeze mode */
  if ( (CpvAccess(lastBreakPointMsg) != NULL) && (CpvAccess(lastBreakPointObject) != NULL) ) {
    EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CpvAccess(lastBreakPointIndex));
    if (breakPointEntryInfo != NULL) {
      breakPointEntryInfo->call(CpvAccess(lastBreakPointMsg), CpvAccess(lastBreakPointObject));
    }
    CpvAccess(lastBreakPointMsg) = NULL;
    CpvAccess(lastBreakPointObject) = NULL;
  }
  else {
    // we were not stopped at a breakpoint, then deliver the first message in the debug queue
    if (!CdsFifo_Empty(CpvAccess(debugQueue))) {
      CpvAccess(skipBreakpoint) = 1;
      char *queuedMsg = (char *)CdsFifo_Dequeue(CpvAccess(debugQueue));
      CmiHandleMessage(queuedMsg);
      CpvAccess(skipBreakpoint) = 0;
    }
  }
}

//ccs handler when continue from a break point
extern "C"
void CpdContinueFromBreakPoint ()
{
    CpdUnFreeze();
    if ( (CpvAccess(lastBreakPointMsg) != NULL) && (CpvAccess(lastBreakPointObject) != NULL) )
    {
        EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CpvAccess(lastBreakPointIndex));
        if (breakPointEntryInfo != NULL)
           breakPointEntryInfo->call(CpvAccess(lastBreakPointMsg), CpvAccess(lastBreakPointObject));
    }
    CpvAccess(lastBreakPointMsg) = NULL;
    CpvAccess(lastBreakPointObject) = NULL;
}

//ccs handler to set a breakpoint with entry function name msg
void CpdSetBreakPoint (char *msg)
{
  char functionName[128];
  int tableSize, tableIdx = 0;
  sscanf(msg+CmiMsgHeaderSizeBytes, "%s", functionName);
  if (strlen(functionName) > 0)
  {
    tableSize = _entryTable.size();
    // Replace entry in entry table with _call_freeze_on_break_point
    // retrieve epIdx for entry method
    //for (tableIdx=0; tableIdx < tableSize; tableIdx++)
    //{
       //if (strstr(_entryTable[tableIdx]->name, functionName) != NULL)
       //{
    tableIdx = atoi(functionName);
    if (tableIdx < 0 || tableIdx >= tableSize) {
      CmiPrintf("[ERROR]Entrypoint was not found for function %s\n", functionName);
      return;
    }
            EntryInfo * breakPointEntryInfo = new EntryInfo(_entryTable[tableIdx]->name, _entryTable[tableIdx]->call, _entryTable[tableIdx]->msgIdx, _entryTable[tableIdx]->chareIdx );
           CmiPrintf("Breakpoint is set for function %s with an epIdx = %ld\n", _entryTable[tableIdx]->name, tableIdx);
           CpvAccess(breakPointEntryTable)->put(tableIdx) = breakPointEntryInfo;
           _entryTable[tableIdx]->name = "debug_breakpoint_ep";
           _entryTable[tableIdx]->call = (CkCallFnPtr)_call_freeze_on_break_point;
           _entryTable[tableIdx]->msgIdx = CpvAccess(_debugMsg);
           _entryTable[tableIdx]->chareIdx = CpvAccess(_debugChare);
           //_debugEntryTable[tableIdx].isBreakpoint = CmiTrue;
           //break;
       //}
    //}
    //if (tableIdx == tableSize)
    //{
    //  CmiPrintf("[ERROR]Entrypoint was not found for function %s\n", functionName);
    //  return;
    //}

  }

}

void CpdQuitDebug()
{
  CpdContinueFromBreakPoint();
  CkExit();
}

void CpdRemoveBreakPoint (char *msg)
{
  char functionName[128];
  sscanf(msg+CmiMsgHeaderSizeBytes, "%s", functionName);
  if (strlen(functionName) > 0) {
    int idx = atoi(functionName);
    if (idx < 0 || idx >= _entryTable.size()) {
      CmiPrintf("[ERROR]Entrypoint was not found for function %s\n", functionName);
      return;
    }
    //void *objPointer;
    //void *keyPointer;
    //CkHashtableIterator *it = CpvAccess(breakPointEntryTable)->iterator();
    //while(NULL!=(objPointer = it->next(&keyPointer)))
    //{
    //EntryInfo * breakPointEntryInfo = *(EntryInfo **)objPointer;
    EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(idx);
    if (breakPointEntryInfo != NULL) {
      _entryTable[idx]->name =  breakPointEntryInfo->name;
      _entryTable[idx]->call = (CkCallFnPtr)breakPointEntryInfo->call;
      _entryTable[idx]->msgIdx = breakPointEntryInfo->msgIdx;
      _entryTable[idx]->chareIdx = breakPointEntryInfo->chareIdx;
      //_debugEntryTable[idx].isBreakpoint = CmiFalse;
      CmiPrintf("Breakpoint is removed for function %s with epIdx %ld\n", _entryTable[idx]->name, idx);
      CpvAccess(breakPointEntryTable)->remove(idx);
    }
  }
}

void CpdRemoveAllBreakPoints ()
{
  //all breakpoints removed
  void *objPointer;
  void *keyPointer;
  CkHashtableIterator *it = CpvAccess(breakPointEntryTable)->iterator();
  while(NULL!=(objPointer = it->next(&keyPointer)))
  {
    EntryInfo * breakPointEntryInfo = *(EntryInfo **)objPointer;
    int idx = *(int *)keyPointer;
    _entryTable[idx]->name =  breakPointEntryInfo->name;
    _entryTable[idx]->call = (CkCallFnPtr)breakPointEntryInfo->call;
    _entryTable[idx]->msgIdx = breakPointEntryInfo->msgIdx;
    _entryTable[idx]->chareIdx = breakPointEntryInfo->chareIdx;
  }
}

extern "C" int CpdIsCharmDebugMessage(void *msg) {
  envelope *env = (envelope*)msg;
  // Later should use "isDebug" value, but for now just bypass all intrinsic EPs
  return CmiGetHandler(msg) != _charmHandlerIdx || env->getMsgtype() == ForVidMsg ||
         env->getMsgtype() == FillVidMsg || _entryTable[env->getEpIdx()]->inCharm;
}



CpvExtern(char *, displayArgument);

void CpdStartGdb(void)
{
#if !defined(_WIN32) || defined(__CYGWIN__)
  FILE *f;
  char gdbScript[200];
  int pid;
  if (CpvAccess(displayArgument) != NULL)
  {
     /*CmiPrintf("MY NODE IS %d  and process id is %d\n", CmiMyPe(), getpid());*/
     sprintf(gdbScript, "/tmp/cpdstartgdb.%d.%d", getpid(), CmiMyPe());
     f = fopen(gdbScript, "w");
     fprintf(f,"#!/bin/sh\n");
     fprintf(f,"cat > /tmp/start_gdb.$$ << END_OF_SCRIPT\n");
     fprintf(f,"shell /bin/rm -f /tmp/start_gdb.$$\n");
     //fprintf(f,"handle SIGPIPE nostop noprint\n");
     fprintf(f,"handle SIGWINCH nostop noprint\n");
     fprintf(f,"handle SIGWAITING nostop noprint\n");
     fprintf(f, "attach %d\n", getpid());
     fprintf(f,"END_OF_SCRIPT\n");
     fprintf(f, "DISPLAY='%s';export DISPLAY\n",CpvAccess(displayArgument));
     fprintf(f,"/usr/X11R6/bin/xterm ");
     fprintf(f," -title 'Node %d ' ",CmiMyPe());
     fprintf(f," -e /usr/bin/gdb -x /tmp/start_gdb.$$ \n");
     fprintf(f, "exit 0\n");
     fclose(f);
     if( -1 == chmod(gdbScript, 0755))
     {
        CmiPrintf("ERROR> chmod on script failed!\n");
        return;
     }
     pid = fork();
     if (pid < 0)
        { perror("ERROR> forking to run debugger script\n"); exit(1); }
     if (pid == 0)
     {
         //CmiPrintf("In child process to start script %s\n", gdbScript);
 if (-1 == execvp(gdbScript, NULL))
            CmiPrintf ("Error> Could not Execute Debugger Script: %s\n",strerror
(errno));

      }
    }
#endif
}

extern "C" {
  size_t cpd_memory_length(void*);
  void cpd_memory_pup(void*,void*,CpdListItemsRequest*);
  void cpd_memory_leak(void*,void*,CpdListItemsRequest*);
  size_t cpd_memory_getLength(void*);
  void cpd_memory_get(void*,void*,CpdListItemsRequest*);
}


void CpdCharmInit()
{
  CpdListRegister(new CpdListAccessor_c("converse/memory",cpd_memory_length,0,cpd_memory_pup,0));
  //CpdListRegister(new CpdListAccessor_c("converse/memory/leak",cpd_memory_length,0,cpd_memory_leak,0));
  CpdListRegister(new CpdListAccessor_c("converse/memory/data",cpd_memory_getLength,0,cpd_memory_get,0,false));

  CpdBreakPointInit();
  CcsRegisterHandler("ccs_set_break_point",(CmiHandler)CpdSetBreakPoint);
  CcsRegisterHandler("ccs_remove_break_point",(CmiHandler)CpdRemoveBreakPoint);
  CcsRegisterHandler("ccs_remove_all_break_points",(CmiHandler)CpdRemoveAllBreakPoints);
  CcsRegisterHandler("ccs_continue_break_point",(CmiHandler)CpdContinueFromBreakPoint);
  CcsRegisterHandler("ccs_single_step",(CmiHandler)CpdDeliverSingleMessage);
  CcsRegisterHandler("ccs_debug_quit",(CmiHandler)CpdQuitDebug);
  CcsRegisterHandler("ccs_debug_startgdb",(CmiHandler)CpdStartGdb);
  CpdListRegister(new CpdListAccessor_c("hostinfo",hostInfoLength,0,hostInfo,0));
  CpdListRegister(new CpdList_localQ());
  CpdListRegister(new CpdList_arrayElementNames());
  CpdListRegister(new CpdList_arrayElements());
  CpdListRegister(new CpdList_objectNames());
  CpdListRegister(new CpdList_object());
  CpdIsDebugMessage = CpdIsCharmDebugMessage;
}

#else

void CpdCharmInit()
{
}

void CpdFinishInitialization() {
}

void CpdBeforeEp(int ep, void *obj) {
}

void CpdAfterEp(int ep) {
}

#endif /*CMK_CCS_AVAILABLE*/
