/// Modest statistics gathering facility for POSE
#include "charm++.h"
#include "stats.h"
#include "stats.def.h"

CkChareID theGlobalStats;
CkGroupID theLocalStats;

extern void POSE_exit();

/// Send local stats to global collector
void localStat::SendStats()
{
  CProxy_globalStat gstat(theGlobalStats);
  localStatSummary *m = new localStatSummary;
  m->doTime = totalTime;
  m->rbTime = rollbackTime;
  m->gvtTime = gvtTime;
  m->simTime = simTime;
  m->cpTime = cpTime;
  m->canTime = canTime;
  m->lbTime = lbTime;
  m->fcTime = fcTime;
  m->commTime = commTime;
  m->cpBytes = cpBytes;
  m->maxDo = maxDo;
  m->minDo = minDo;
  m->pe = CkMyPe();
  m->dos = dos;
  m->undos = undos;
  m->commits = commits;
  m->loops = loops;
  m->gvts = gvts;
  m->maxChkPts = maxChkPts;
  m->maxGVT = maxGVT;
  m->maxGRT = maxGRT;
  gstat.localStatReport(m);
}

// Basic Constructor
globalStat::globalStat(void)
{
  doAvg = doMax = rbAvg = rbMax = gvtAvg = gvtMax = simAvg = simMax = 
    cpAvg = cpMax = canAvg = canMax = lbAvg = lbMax = fcAvg = fcMax = 
    commAvg = commMax =
    maxTime = maxDo = minDo = avgDo = GvtTime = maxGRT = 0.0; 
  cpBytes = reporting = totalDos = totalUndos = totalCommits = totalLoops = 
    totalGvts = maxChkPts = maxGVT = 0;
}

// Receive, calculate and print statistics
void globalStat::localStatReport(localStatSummary *m)
{
  double tmpMax;
  // accumulate data from local stats collectors
  totalDos += m->dos;
  totalUndos += m->undos;
  totalCommits += m->commits;
  totalLoops += m->loops;
  totalGvts += m->gvts;
  maxChkPts += m->maxChkPts;
  doAvg += m->doTime;
  if (maxGRT < m->maxGRT) maxGRT = m->maxGRT;
  if (maxGVT < m->maxGVT) maxGVT = m->maxGVT;
  if (m->maxDo > maxDo)
    maxDo = m->maxDo;
  if ((minDo < 0.0) || (m->minDo < minDo))
    minDo = m->minDo;
  if (m->doTime > doMax)
    doMax = m->doTime;
  rbAvg += m->rbTime;
  if (m->rbTime > rbMax)
    rbMax = m->rbTime;
  gvtAvg += m->gvtTime;
  if (m->gvtTime > gvtMax)
    gvtMax = m->gvtTime;
  simAvg += m->simTime;
  if (m->simTime > simMax)
    simMax = m->simTime;
  cpAvg += m->cpTime;
  if (m->cpTime > cpMax)
    cpMax = m->cpTime;
  canAvg += m->canTime;
  if (m->canTime > canMax)
    canMax = m->canTime;
  lbAvg += m->lbTime;
  if (m->lbTime > lbMax)
    lbMax = m->lbTime;
  fcAvg += m->fcTime;
  if (m->fcTime > fcMax)
    fcMax = m->fcTime;
  commAvg += m->commTime;
  if (m->commTime > commMax)
    commMax = m->commTime;
  cpBytes += m->cpBytes;
  tmpMax = m->doTime + m->rbTime + m->gvtTime + m->simTime + m->cpTime 
    + m->canTime + m->lbTime + m->fcTime + m->commTime;
  CkFreeMsg(m);
  if (tmpMax > maxTime)
    maxTime = tmpMax;
  reporting++;

#ifdef SEQUENTIAL_POSE
  totalLoops = totalDos;
  totalGvts = 1;
#endif
  CkAssert(totalGvts > 0);
  CkAssert(totalDos > 0);
  CkAssert(totalLoops > 0);
  GvtTime = gvtAvg/totalGvts;
  if (reporting == CkNumPes()) { //all local stats are in; compute final values
    avgDo = doAvg / totalDos;
    doAvg /= CkNumPes();
    rbAvg /= CkNumPes();
    gvtAvg /= CkNumPes();
    simAvg /= CkNumPes();
    cpAvg /= CkNumPes();
    canAvg /= CkNumPes();
    lbAvg /= CkNumPes();
    fcAvg /= CkNumPes();
    commAvg /= CkNumPes();
    maxChkPts /= CkNumPes();
    // print stats table (all one print to avoid breaking up)
    CkPrintf("------------------------------------------------------------------------------------\n   | DO     | RB     | GVT    | SIM    | CP     | CAN    | LB     | FC     | COMM   |\n---|--------|--------|--------|--------|--------|--------|--------|--------|--------|\nmax| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f|\navg| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f| %7.2f|\n------------------------------------------------------------------------------------\nMax time on a PE:%7.2f, Speculative Events:%d Actual Events:%d Commits:%d\nGRAINSIZE INFO: Avg: %10.6f Max: %10.6f Min: %10.6f\nGVT iterations=%d  Avg time per iteration=%f\ntotalLoops=%d effectiveGS=%10.6f\n", 
	     doMax, rbMax, gvtMax, simMax, cpMax, canMax, lbMax, fcMax, commMax, doAvg, rbAvg, gvtAvg, simAvg, cpAvg, canAvg, lbAvg, fcAvg, commAvg, maxTime, totalDos, totalDos-totalUndos, totalCommits, avgDo, maxDo, minDo, totalGvts, GvtTime, totalLoops, (doAvg*CkNumPes())/totalLoops);
    //CkPrintf("Avg. Max# Checkpoints=%d Bytes checkpointed=%d\n", maxChkPts, cpBytes);

    if(pose_config.dop){
      CkPrintf("Overhead-free maximally parallel runtime=%f  Max GVT=%d\n",
	     maxGRT, maxGVT);
      DOPcalc(maxGVT, maxGRT);
    }

    POSE_exit();
    /*
    reporting = 0;
    doAvg = doMax = rbAvg = rbMax = gvtAvg = gvtMax = simAvg = simMax = 
      cpAvg = cpMax = canAvg = canMax = lbAvg = lbMax = fcAvg = fcMax = 
      commAvg, commMax, maxTime = maxDo = minDo = avgDo = GvtTime = 0.0; 
    cpBytes = reporting = totalDos = totalUndos = totalCommits = totalLoops = 
      totalGvts = maxChkPts = 0;
    */
  }
}


void globalStat::DOPcalc(int gvt, double grt)
{
  int vinStart, vinEnd, gvtp = gvt + 1, i;
  double rinStart, rinEnd;
  FILE *fp;
  char filename[20], line[80];
  unsigned short int *gvtDOP, *grtDOP;
  unsigned long int grtp = (unsigned long int)(grt*1000000.0) + 1, j, 
    usStart, usEnd;
  int modelPEs=0, simulationPEs=0;

  CkPrintf("Generating DOP measures...\n");
  gvtDOP = (unsigned short int *)malloc(gvtp*sizeof(unsigned short int));
  grtDOP = (unsigned short int *)malloc(grtp*sizeof(unsigned short int));
  for (i=0; i<gvtp; i++) gvtDOP[i] = 0;
  for (i=0; i<grtp; i++) grtDOP[i] = 0;
  for (i=0; i<CkNumPes(); i++) { // read each processor's log
    sprintf(filename, "dop%d.log\0", i);
    fp = fopen(filename, "r");
    if (!fp) {
      CkPrintf("Cannot open file %s... exiting.\n", filename);
      POSE_exit();
    }
    CkPrintf("Reading file %s...\n", filename);
    while (fgets(line, 80, fp)) {
      if (sscanf(line, "%lf %lf %d %d\n", &rinStart, &rinEnd, 
		 &vinStart, &vinEnd) == 4) {
	usStart = (unsigned long int)(rinStart * 1000000.0);
	usEnd = (unsigned long int)(rinEnd * 1000000.0);
	for (j=usStart; j<usEnd; j++) grtDOP[j]++;
	if (usStart == usEnd) grtDOP[usStart]++;
	if (vinStart > -1)
	  for (j=vinStart; j<=vinEnd; j++) gvtDOP[j]++;
      }
      else CkPrintf("WARNING: DOP post-processing corrupted... likely NFS to blame.\n");
    }
    fclose(fp);
  }
  int avgPEs = 0;
  int zed = 0;
  fp = fopen("dop_mod.out", "w");
  for (i=0; i<gvtp; i++) {
    if ((gvtDOP[i] != 0) || (zed == 0))
      fprintf(fp, "%d %d\n", i, gvtDOP[i]);
    if (gvtDOP[i] == 0) zed = 1;
    else zed = 0;
    avgPEs += gvtDOP[i];
    if (gvtDOP[i] > modelPEs) modelPEs = gvtDOP[i];
  }
  avgPEs /= gvtp;
  fclose(fp);
  fp = fopen("dop_sim.out", "w");
  for (i=0; i<grtp; i++) {
    fprintf(fp, "%d %d\n", i, grtDOP[i]);
    if (grtDOP[i] > simulationPEs) simulationPEs = grtDOP[i];
  } 
  fclose(fp);
  CkPrintf("Max model PEs: %d  Max simulation PEs: %d  Recommended #PEs: %d\n",
	   modelPEs, simulationPEs, avgPEs);
}
