package projections.analysis;

/*
LogReader.java: Charm++ projections.  
Converted to Java by Theckla Louchios
Improved by Orion Sky Lawlor, olawlor@acm.org

Reads .log files generated by a charm++ program compiled with
"-tracemode projections".  Converts .log files into 
gigantic multidimentional arrays indexed by entry point,
processor, and time interval.
*/
import java.lang.*;
import java.io.*;
import projections.misc.*;
import java.util.*;

public class LogReader extends ProjDefs
{
	public static final int SYS_Q=0; //sysUsgData[SYS_Q] is length of message queue
	public static final int SYS_CPU=1; //sysUsgData[SYS_CPU] is percent processing time
	public static final int SYS_IDLE=2; //sysUsgData[SYS_IDLE] is percent idle time
	public static final int IDLE_ENTRY=-1; //Magic entry method (indicates "idle")
	public static final int CREATE=0; //Number of creations bin
	public static final int PROCESS=1; //Number of invocations bin
	public static final int TIME=2; //Time (us) spent processing bin

	private int[][][] sysUsgData;
	private int[][][][] userEntries;
	private int[][][][] categorized;
	private long progStartTime;
	private int numProcessors;
	private int numUserEntries; //Number of user entry points
	private long startTime; //Time the current entry method started
	private int currentEntry; //Currently executing entry method
	private int currentMtype; //Current message type
	private int interval;//Current interval number
	private int curPe;//Current source processor #
	private int numIntervals;//Total number of intervals
	private long intervalSize;//Length of an interval (us)
	private int processing;
	private boolean byEntryPoint;
	
	//Add the given amount of time to the current entry for interval j
	final private void addToInterval(int extra,int j,boolean maxPercents)
	{
	if (processing<=0) return; //Not processing at all
	if (currentEntry == IDLE_ENTRY) //Idle time
		sysUsgData[SYS_IDLE][curPe][j] += maxPercents?100:extra;
	else { //Processing an entry method
	//System.out.println("Adding "+extra+"us for #"+curPe+":"+j);
		sysUsgData[SYS_CPU][curPe][j] += maxPercents?100:extra;
		if (byEntryPoint) {
			userEntries[currentEntry][2][curPe][j] += extra; 
			int catIdx=mtypeToCategoryIdx(currentMtype); 
			if (catIdx!=-1) categorized[catIdx][TIME][curPe][j]+=extra;
		}
	}
	startTime+=extra;
	}
	//Add this entry point to the counts
	final private void count(int mtype,int entry,int TYPE)
	{
		if (!byEntryPoint) return;
	if (userEntries[entry][TYPE]==null) {
		userEntries[entry][TYPE]=new int[numProcessors][numIntervals];
		if (TYPE==PROCESS) //Add a time array, too
			userEntries[entry][TIME]=new int[numProcessors][numIntervals];
	}
	userEntries[entry][TYPE][curPe][interval]++;
	
	int catIdx=mtypeToCategoryIdx(mtype);
	if (catIdx!=-1) {
		if (categorized[catIdx][TYPE]==null) {
			categorized[catIdx][TYPE]=new int[numProcessors][numIntervals];
			if (TYPE==PROCESS) //Add a time array, too
			  categorized[catIdx][TIME]=new int[numProcessors][numIntervals];
		}
		categorized[catIdx][TYPE][curPe][interval]++;
	}
	}
	//This is called when a log entry crosses a timing interval boundary
	private void fillToInterval(int newinterval)
	{
	if (interval>=newinterval) return; //Nothing to do in this case
	
	//Finish off the current interval
	int extra = (int) ((interval+1)*intervalSize - startTime);
	addToInterval(extra,interval,false);

	//Convert system usage and idle time from us to percent of an interval
	rescale(interval);

	//Fill in any intervals we would skip over completely
	for(int j=interval+1; j<newinterval; j++)
		addToInterval((int)intervalSize,j,true);

	interval = newinterval;
	}
	public int[][][][] getSystemMsgs()
	{ return categorized; }
	public int[][][] getSystemUsageData()
	{ return sysUsgData; }
	public int[][][][] getUserEntries()
	{ return userEntries; }
	private void intervalCalc(int type, int mtype, int entry, long time) throws IOException
	{
/*	if (entry < 0 || entry >= numUserEntries)
	{
	  System.out.println("Error: Invalid entry number "+entry+" in log file");   
	  throw new IOException();
	} */
	fillToInterval((int)(time/intervalSize));
	switch(type) {
	case ENQUEUE:
		sysUsgData[SYS_Q][curPe][interval]++;//Lengthen system queue
		break;
		case CREATION:
		sysUsgData[SYS_Q][curPe][interval]++;//Lengthen system queue
		count(mtype,entry,CREATE);
		break;
		case BEGIN_PROCESSING:
		processing++;
		startTime = time;
		sysUsgData[SYS_Q][curPe][interval]--;//Shorten system queue
		count(currentMtype = mtype,currentEntry = entry,PROCESS);
		break;
		case END_PROCESSING:
		addToInterval((int)(time-startTime),interval,false);
		processing--;
		break;
		case BEGIN_IDLE:
		processing++;
		startTime = time;
		currentEntry = IDLE_ENTRY;
		break;
		case END_IDLE:
		addToInterval((int)(time-startTime),interval,false);
		processing--;
		break;
	default:
		System.out.println("Unhandled type "+type+" in logreader!");
		break;
	}
	}
//Maps a message type to a categorized system message number
	final private int mtypeToCategoryIdx(int mtype) {
		switch(mtype) {
		case NEW_CHARE_MSG: return 0;
		case FOR_CHARE_MSG: return 1;
		case BOC_INIT_MSG: return 2;
		case LDB_MSG: return 3;
		case QD_BROADCAST_BOC_MSG: case QD_BOC_MSG:
			return 4;
		default:
			return -1;
		}
	}
	public void read(StsReader sts,
		long totalTime, long reqIntervalSize,
		boolean NbyEntryPoint)
	{
	int type;
	int mtype;
	long time;
		int entry;
	int event;
	int pe;
	
	numProcessors = sts.getProcessorCount();
	numUserEntries = sts.getEntryCount();
	intervalSize=reqIntervalSize;
	numIntervals = (int) (totalTime / (double)intervalSize + 1.0);
	byEntryPoint=NbyEntryPoint;

	ProgressDialog bar=new ProgressDialog("Reading log files...");
	double allocEffort=0.5;//Number of logs the allocations are worth
	double totalEffort=allocEffort+sts.getProcessorCount();
	bar.progress(0,"allocating");
	sysUsgData = new int[3][numProcessors][numIntervals];
	if (byEntryPoint) {
		userEntries = new int[numUserEntries][3][][];
		categorized = new int[5][3][][];
	}
	int nPe=sts.getProcessorCount();
	for (curPe = 0; curPe <nPe; curPe++)
		try {
		processing = 0;
	        interval = 0;
	        currentEntry = -1;
	        startTime =0;
		FileReader file = new FileReader(sts.getLogName(curPe));
		AsciiIntegerReader log=new AsciiIntegerReader(new BufferedReader(file));
		log.nextLine(); // The first line contains junk
		//The second line gives the program start time
		log.nextInt();
		progStartTime = 0;//log.nextLong();
		int nLines=2;
		if (!bar.progress(allocEffort+curPe,totalEffort,"Loading "+(curPe+1)+" of "+nPe))
			break;//User cancelled load
		
		try { while (true) { //EOFException will terminate loop
			log.nextLine();//Skip any junk from previous line
			type = log.nextInt();
		  	nLines++;
			switch (type) {
			case BEGIN_IDLE: case END_IDLE:
				time = log.nextLong();
				pe = log.nextInt();
								intervalCalc(type, 0, 0, (time-progStartTime));
				break;
			case BEGIN_PROCESSING: case END_PROCESSING:
			case CREATION:
				mtype = log.nextInt();
				entry = log.nextInt();
				time = log.nextLong();
				event = log.nextInt();
				pe = log.nextInt();
			//System.out.println(type+" "+mtype+" "+entry+" "+time);
								intervalCalc(type, mtype, entry, (time-progStartTime));
				break;
			case ENQUEUE:
				mtype = log.nextInt();
				time = log.nextLong();
				event = log.nextInt();
				pe = log.nextInt();
								intervalCalc(type, mtype, 0, (time-progStartTime));
				break;
			case END_COMPUTATION:
				time = log.nextLong();
								fillToInterval(numIntervals);
 				break;
 			default:
 				break;//Just skip this line
 			}
		}} catch (EOFException e) {
			log.close();
		}
	    }
	catch (IOException e)
	   { System.out.println("Exception reading log file #"+curPe); return;}
	bar.done();
	}
	//Convert system usage and idle time from us to percent of an interval
	private void rescale(int j)
	{
	sysUsgData[SYS_CPU ][curPe][j]=
		(int)(sysUsgData[SYS_CPU ][curPe][j]*100/intervalSize);
	sysUsgData[SYS_IDLE][curPe][j]=
		(int)(sysUsgData[SYS_IDLE][curPe][j]*100/intervalSize);
	}
}