/*****************************************************************************
 * A few useful built-in CPD and CCS handlers.
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>		// for chmod

#include "converse.h"
#include "ckhashtable.h"
#include "conv-ccs.h"
#include "debug-charm.h"
#include "sockRoutines.h"
#include "charm.h"
#include "middle.h"
#include "cklists.h"
#include "register.h"
//#include "queueing.h"

#if CMK_CCS_AVAILABLE

#include "ck.h"


/************ Array Element CPD Lists ****************/

/**
  Count array elements going by until they reach this 
  range (lo to hi), then start passing them to dest.
*/
template <class T>
class CkArrayElementRangeIterator : public CkLocIterator {
private:
   T *dest;
   CkArray *mgr;
   int cur,lo,hi;
public:
   CkArrayElementRangeIterator(T *dest_,int l,int h) 
   	:dest(dest_),mgr(0),cur(0),lo(l),hi(h) {}
   
   /** Call add for every in-range array element on this processor */
   void iterate(void)
   { /* Walk the groupTable for arrays (FIXME: get rid of _groupIDTable) */
     int numGroups=CkpvAccess(_groupIDTable)->size();
     for(int i=0;i<numGroups;i++) {
        IrrGroup *obj = CkpvAccess(_groupTable)->find((*CkpvAccess(_groupIDTable))[i]).getObj();
	if (obj->isArrMgr()) 
	{ /* This is an array manager: examine its array elements */
	  mgr=(CkArray *)obj;
	  mgr->getLocMgr()->iterate(*this);
	}
     }
   }
   
   // Called by location manager's iterate function
   virtual void addLocation (CkLocation &loc)
   {
     if (cur>=lo && cur<hi) 
     { /* This element is in our range-- look it up */
       dest->add(cur,mgr->lookup(loc.getIndex()));
     }
     cur++;
   }
   
   // Return the number of total array elements seen so far.
   int getCount(void) {return cur;}
};

class ignoreAdd {
public: void add(int cur,ArrayElement *elt) {}
};

/** Coarse: examine array element names */
class CpdList_arrayElementNames : public CpdListAccessor {
  PUP::er *pp; // Only used while inside pup routine.
public:
  virtual const char * getPath(void) const {return "charm/arrayElementNames";}
  virtual int getLength(void) const {
    CkArrayElementRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    CkArrayElementRangeIterator<CpdList_arrayElementNames> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements
  }
  void add(int cur,ArrayElement *elt) 
  { // Just grab the name and nothing else:
         PUP::er &p=*pp;
	 beginItem(p,cur);
         p.comment("name");
	 char *n=elt->ckDebugChareName();
	 p(n,strlen(n));
	 free(n);
  }
};

/** Detailed: examine array element data */
class CpdList_arrayElements : public CpdListAccessor {
  PUP::er *pp; // Only used while inside pup routine.
public:
  virtual const char * getPath(void) const {return "charm/arrayElements";}
  virtual int getLength(void) const {
    CkArrayElementRangeIterator<ignoreAdd> it(0,0,0);
    it.iterate();
    return it.getCount();
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    pp=&p;
    CkArrayElementRangeIterator<CpdList_arrayElements> it(this,req.lo,req.hi);
    it.iterate(); // calls "add" for in-range elements
  }
  void add(int cur,ArrayElement *elt) 
  { // Pup the element data
         PUP::er &p=*pp;
	 beginItem(p,cur);
	 elt->ckDebugPup(p);
  }
};

/************ Message CPD Lists ****************/
CpvCExtern(void *,debugQueue);


// Interpret data in a message in a user-friendly way.
//  Ignores most of the envelope fields used by CkPupMessage,
//  and instead concentrates on user data
void CpdPupMessage(PUP::er &p, void *msg)
{
  envelope *env=UsrToEnv(msg);
  int wasPacked=env->isPacked();
  int size=env->getTotalsize();
  int prioBits=env->getPriobits();
  PUPn(wasPacked);
  PUPn(prioBits);
  int userSize=size-sizeof(envelope)-sizeof(int)*CkPriobitsToInts(prioBits);
  PUPn(userSize);
  
  p.synchronize(PUP::sync_last_system);
  
  int ep=CkMessageToEpIdx(msg);
  PUPn(ep);
  
  if (_entryTable[ep]->messagePup!=NULL) 
    _entryTable[ep]->messagePup(p,msg);
  else
    CkMessage::ckDebugPup(p,msg);
}

//Cpd Lists for local and scheduler queues
class CpdList_localQ : public CpdListAccessor {

public:
  CpdList_localQ() {}
  virtual const char * getPath(void) const {return "converse/localqueue";}
  virtual int getLength(void) const {
    int x = CdsFifo_Length((CdsFifo)(CpvAccess(debugQueue)));
    //CmiPrintf("*******Returning fifo length %d*********\n", x);
    //return CdsFifo_Length((CdsFifo)(CpvAccess(CmiLocalQueue)));
    return x;
  }
  virtual void pup(PUP::er &p, CpdListItemsRequest &req) {
    int length = CdsFifo_Length((CdsFifo)(CpvAccess(debugQueue)));
    void ** messages = CdsFifo_Enumerate(CpvAccess(debugQueue));
    int curObj=0;
    
    for(curObj=req.lo; curObj<req.hi; curObj++)
    if ((curObj>=0) && (curObj<length))
    {
        beginItem(p,curObj);
        void *msg=messages[curObj]; /* converse message */
	int isCharm=0;
        const char *type="Converse";
        p.comment("name");
        char name[128];
	if (CmiGetHandler(msg)==_charmHandlerIdx) {isCharm=1; type="Local Charm";}
	if (CmiGetXHandler(msg)==_charmHandlerIdx) {isCharm=1; type="Network Charm";}
        sprintf(name,"%s %d: %s (%d)","Message",curObj,type,CmiGetHandler(msg));
        p(name, strlen(name));
	
	if (isCharm) 
	{ /* charm message */
	  p.synchronize(PUP::sync_begin_object);
	  envelope *env=(envelope *)msg;
	  CkUnpackMessage(&env);
	  messages[curObj]=env;
          CpdPupMessage(p, EnvToUsr(env));
          //CkPupMessage(p, &messages[curObj], 0);
	  p.synchronize(PUP::sync_end_object);
	}
    }

  }
};


/****************** Breakpoints and other debug support **************/

typedef CkHashtableTslow<int,EntryInfo *> CpdBpFuncTable_t;


extern void CpdFreeze(void);
extern void CpdUnFreeze(void);


CpvStaticDeclare(int, _debugMsg);
CpvStaticDeclare(int, _debugChare);

CpvStaticDeclare(CpdBpFuncTable_t *, breakPointEntryTable);

CpvStaticDeclare(void *, lastBreakPointMsg);
CpvStaticDeclare(void *, lastBreakPointObject);
CpvStaticDeclare(int, lastBreakPointIndex);

void CpdBreakPointInit()
{
  CpvInitialize(void *, lastBreakPointMsg);
  CpvInitialize(void *, lastBreakPointObject);
  CpvInitialize(int, lastBreakPointIndex);
  CpvInitialize(int, _debugMsg);
  CpvInitialize(int, _debugChare);
  CpvInitialize(CpdBpFuncTable_t *, breakPointEntryTable);
  CpvAccess(lastBreakPointMsg) = NULL;
  CpvAccess(lastBreakPointObject) = NULL;
  CpvAccess(lastBreakPointIndex) = 0;
  CpvAccess(_debugMsg) = CkRegisterMsg("debug_msg",0,0,0);
  CpvAccess(_debugChare) = CkRegisterChare("debug_Chare",0);
  CpvAccess(breakPointEntryTable) = new CpdBpFuncTable_t(10,0.5,CkHashFunction_int,CkHashCompare_int );
}



static void _call_freeze_on_break_point(void * msg, void * object)
{
      //Save breakpoint entry point index. This is retrieved from msg.
      //So that the appropriate EntryInfo can be later retrieved from the hash table 
      //of break point function entries, on continue.
      CpvAccess(lastBreakPointMsg) = msg;
      CpvAccess(lastBreakPointObject) = object;
      CpvAccess(lastBreakPointIndex) = CkMessageToEpIdx(msg);
      EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CpvAccess(lastBreakPointIndex));
      CmiPrintf("Break point reached for Function = %s\n", breakPointEntryInfo->name);
      CpdFreeze();
}



//ccs handler when continue from a break point
extern "C"
void CpdContinueFromBreakPoint ()
{
    CpdUnFreeze();
    if ( (CpvAccess(lastBreakPointMsg) != NULL) && (CpvAccess(lastBreakPointObject) != NULL) )
    {
        EntryInfo * breakPointEntryInfo = CpvAccess(breakPointEntryTable)->get(CpvAccess(lastBreakPointIndex));
        if (breakPointEntryInfo != NULL)
           breakPointEntryInfo->call(CpvAccess(lastBreakPointMsg), CpvAccess(lastBreakPointObject));
    }
    CpvAccess(lastBreakPointMsg) = NULL;
    CpvAccess(lastBreakPointObject) = NULL;
}

//ccs handler to set a breakpoint with entry function name msg
void CpdSetBreakPoint (char *msg)
{
  char functionName[128];
  int tableSize, tableIdx = 0;
  sscanf(msg+CmiMsgHeaderSizeBytes, "%s", functionName);
  if (strlen(functionName) > 0)
  {
    tableSize = _entryTable.size();
    // Replace entry in entry table with _call_freeze_on_break_point
    // retrieve epIdx for entry method
    for (tableIdx=0; tableIdx < tableSize; tableIdx++)
    {
       if (strstr(_entryTable[tableIdx]->name, functionName) != NULL)
       {
            EntryInfo * breakPointEntryInfo = new EntryInfo(_entryTable[tableIdx]->name, _entryTable[tableIdx]->call, _entryTable[tableIdx]->msgIdx, _entryTable[tableIdx]->chareIdx );
           CmiPrintf("Breakpoint is set for function %s with an epIdx = %ld\n", _entryTable[tableIdx]->name, tableIdx);
           CpvAccess(breakPointEntryTable)->put(tableIdx) = breakPointEntryInfo;  
           _entryTable[tableIdx]->name = "debug_breakpoint_ep";  
           _entryTable[tableIdx]->call = (CkCallFnPtr)_call_freeze_on_break_point;
           _entryTable[tableIdx]->msgIdx = CpvAccess(_debugMsg); 
           _entryTable[tableIdx]->chareIdx = CpvAccess(_debugChare);
           break;
       }
    }
    if (tableIdx == tableSize)
    {
      CmiPrintf("[ERROR]Entrypoint was not found for function %s\n", functionName); 
      return;
    }

  }

}

void CpdQuitDebug()
{
  CpdContinueFromBreakPoint();
  CkExit();
}

void CpdRemoveBreakPoint (char *msg)
{
  char functionName[128];
  sscanf(msg+CmiMsgHeaderSizeBytes, "%s", functionName);
  void *objPointer;
  void *keyPointer; 
  CkHashtableIterator *it = CpvAccess(breakPointEntryTable)->iterator();
  while(NULL!=(objPointer = it->next(&keyPointer)))
  {
    EntryInfo * breakPointEntryInfo = *(EntryInfo **)objPointer;
    int idx = *(int *)keyPointer;
    if (strstr(breakPointEntryInfo->name, functionName) != NULL){
        _entryTable[idx]->name =  breakPointEntryInfo->name;
        _entryTable[idx]->call = (CkCallFnPtr)breakPointEntryInfo->call;
        _entryTable[idx]->msgIdx = breakPointEntryInfo->msgIdx;
        _entryTable[idx]->chareIdx = breakPointEntryInfo->chareIdx;
        CmiPrintf("Breakpoint is removed for function %s with epIdx %ld\n", _entryTable[idx]->name, idx);
    }
  }
}

void CpdRemoveAllBreakPoints ()
{
  //all breakpoints removed
  void *objPointer;
  void *keyPointer; 
  CkHashtableIterator *it = CpvAccess(breakPointEntryTable)->iterator();
  while(NULL!=(objPointer = it->next(&keyPointer)))
  {
    EntryInfo * breakPointEntryInfo = *(EntryInfo **)objPointer;
    int idx = *(int *)keyPointer;
    _entryTable[idx]->name =  breakPointEntryInfo->name;
    _entryTable[idx]->call = (CkCallFnPtr)breakPointEntryInfo->call;
    _entryTable[idx]->msgIdx = breakPointEntryInfo->msgIdx;
    _entryTable[idx]->chareIdx = breakPointEntryInfo->chareIdx;
  }
}





CpvExtern(char *, displayArgument);

void CpdStartGdb(void)
{
#if !defined(_WIN32) || defined(__CYGWIN__)
  FILE *f;
  char gdbScript[200];
  int pid;
  if (CpvAccess(displayArgument) != NULL)
  {
     /*CmiPrintf("MY NODE IS %d  and process id is %d\n", CmiMyPe(), getpid());*/
     sprintf(gdbScript, "/tmp/cpdstartgdb.%d.%d", getpid(), CmiMyPe());
     f = fopen(gdbScript, "w");
     fprintf(f,"#!/bin/sh\n");
     fprintf(f,"cat > /tmp/start_gdb.$$ << END_OF_SCRIPT\n");
     fprintf(f,"shell /bin/rm -f /tmp/start_gdb.$$\n");
     //fprintf(f,"handle SIGPIPE nostop noprint\n");
     fprintf(f,"handle SIGWINCH nostop noprint\n");
     fprintf(f,"handle SIGWAITING nostop noprint\n");
     fprintf(f, "attach %d\n", getpid());
     fprintf(f,"END_OF_SCRIPT\n");
     fprintf(f, "DISPLAY='%s';export DISPLAY\n",CpvAccess(displayArgument));
     fprintf(f,"/usr/X11R6/bin/xterm ");
     fprintf(f," -title 'Node %d ' ",CmiMyPe());
     fprintf(f," -e /usr/bin/gdb -x /tmp/start_gdb.$$ \n");
     fprintf(f, "exit 0\n");
     fclose(f);
     if( -1 == chmod(gdbScript, 0755))
     {
        CmiPrintf("ERROR> chmod on script failed!\n");
        return;
     }
     pid = fork();
     if (pid < 0)
        { perror("ERROR> forking to run debugger script\n"); exit(1); }
     if (pid == 0)
     {
         //CmiPrintf("In child process to start script %s\n", gdbScript);
 if (-1 == execvp(gdbScript, NULL))
            CmiPrintf ("Error> Could not Execute Debugger Script: %s\n",strerror
(errno));

      }
    }
#endif
}


void CpdCharmInit()
{
  CpdBreakPointInit();
  CcsRegisterHandler("ccs_set_break_point",(CmiHandler)CpdSetBreakPoint);
  CcsRegisterHandler("ccs_remove_break_point",(CmiHandler)CpdRemoveBreakPoint);
  CcsRegisterHandler("ccs_remove_all_break_points",(CmiHandler)CpdRemoveAllBreakPoints);
  CcsRegisterHandler("ccs_continue_break_point",(CmiHandler)CpdContinueFromBreakPoint);
  CcsRegisterHandler("ccs_debug_quit",(CmiHandler)CpdQuitDebug);
  CcsRegisterHandler("ccs_debug_startgdb",(CmiHandler)CpdStartGdb);
  CpdListRegister(new CpdList_localQ());
  CpdListRegister(new CpdList_arrayElementNames());
  CpdListRegister(new CpdList_arrayElements());
}


#endif /*CMK_CCS_AVAILABLE*/












