/*
Very simple demo program showing how to use location manager's
checkpointing support.  

FEATURES
This program:
  -Writes out a checkpoint directory when run with no args,
     using the CkLocIterator interface.
  -Reads in a checkpoint directory when run with (any) arg,
     using the CkLocMgr::resume call.

Should work with any number of array elements, including multidimensional, 
sparse, etc.  Should even work with bound arrays.

LIMITATIONS
This program will only run on one processor-- there is no attempt
to checkpoint or restore array elements that do not live on processor 0.

The checkpoint file is written in machine binary format, and no attempt
is made to allow cross-platform checkpoints; although using the PUP_toNetwork4
or PUP::xlater interfaces this would be fairly easy.

Orion Sky Lawlor, olawlor@acm.org, 1/2/2003
*/
#include <stdio.h>
#include "hello.decl.h"

/*readonly*/ CProxy_Main mainProxy;
/*readonly*/ int nElements;

// Print out an array index to this string as decimal fields
// separated by underscores.
void printIndex(const CkArrayIndex &idx,char *dest) {
	const int *idxData=idx.data();
	for (int i=0;i<idx.nInts;i++) {
		sprintf(dest,"%s%d",i==0?"":"_", idxData[i]);
		dest+=strlen(dest);
	}
}

// Silly demo CkLocIterator: just print the names of each array index encountered:
class IndexPrinter : public CkLocIterator {
	const char *arrayName;
public:
	IndexPrinter(const char *arrayName_) :arrayName(arrayName_) {}
	void addLocation(CkLocation &loc) {
		char idxName[128]; printIndex(loc.getIndex(),idxName);
		printf("Array %s contains index %s\n",arrayName,idxName);
	}
};

// Checkpoint each encountered array location:
class ElementSaver : public CkLocIterator {
	FILE *indexFile; //Output list of array indices and data files
	const char *dirName; //Output directory
public:
	ElementSaver(const char *dirName_) :dirName(dirName_) 
	{
		char indexName[1024];
		sprintf(indexName,"%s/index.txt",dirName);
		indexFile=fopen(indexName,"w");
		if (indexFile==NULL)  CkAbort("Could not create index file");
		fprintf(indexFile,"CHARM++_Checkpoint_File 1.0 %d %d\n",CkMyPe(),CkNumPes());
	}
	~ElementSaver() {
		fclose(indexFile);
	}
	void addLocation(CkLocation &loc) {
		const CkArrayIndex &idx=loc.getIndex();
		const int *idxData=idx.data();
		char idxName[128]; printIndex(idx,idxName);
		char fileName[1024]; sprintf(fileName,"%s.dat",idxName);
		
		//Write a file index entry
		fprintf(indexFile,"%s %d ",fileName,idx.nInts);
		for (int i=0;i<idx.nInts;i++) fprintf(indexFile,"%d ",idxData[i]);
		fprintf(indexFile,"\n");
		
		//Save the actual array element data to the file:
		char pathName[1024];
		sprintf(pathName,"%s/%s",dirName,fileName);
		FILE *f=fopen(pathName,"wb");
		if (!f) CkAbort("Could not create checkpoint file");
		PUP::toDisk p(f);
		loc.pup(p);
		fclose(f);
		
		printf("Saved array index %s to file %s\n",idxName,pathName);
	}
};

// Restore each array location listed in the index file:
class ElementRestorer {
	FILE *indexFile; //Input list of array indices and data files
	const char *dirName; //Input directory
	CkLocMgr *dest; //Place to put new array elements
public:
	ElementRestorer(const char *dirName_,CkLocMgr *dest_) 
		:dirName(dirName_), dest(dest_)
	{
		char indexName[1024];
		sprintf(indexName,"%s/index.txt",dirName);
		indexFile=fopen(indexName,"r");
		if (indexFile==NULL)  CkAbort("Could not read index file");
		char ignored[128]; double version; int srcPE; int srcSize;
		if (4!=fscanf(indexFile,"%s%lf%d%d",ignored,&version,&srcPE,&srcSize))
			CkAbort("Checkpoint index file format error");
		if (version>=2.0) CkAbort("Checkpoint index file format is too new");
	}
	~ElementRestorer() {
		fclose(indexFile);
	}
	
	// Try to restore one array element.  If it worked, return true.
	bool restore(void) {
		// Find the index and filename from the file index:
		CkArrayIndexMax idx;
		char fileName[1024];
		if (fscanf(indexFile,"%s%d",fileName,&idx.nInts)!=2) return false;
		int *idxData=1+((int *)&idx); // YUCK!  Should just use idx.pup!
		for (int i=0;i<idx.nInts;i++) fscanf(indexFile,"%d",&idxData[i]);
		
		//Restore the actual array element data from the file:
		char pathName[1024];
		sprintf(pathName,"%s/%s",dirName,fileName);
		FILE *f=fopen(pathName,"rb");
		if (!f) CkAbort("Could not read checkpoint file");
		PUP::fromDisk p(f);
		dest->resume(idx,p);
		fclose(f);
		
		printf("Restored an index from file %s\n",pathName);
		return true;
	}
};



/*mainchare*/
class Main : public Chare
{
public:
  Main(CkArgMsg* m)
  {
    //Process command-line arguments
    nElements=5;
    bool doRestart=false;
    if(m->argc >1 ) doRestart=true; // gotta be "+restart"
    delete m;

    //Start the computation
    CkPrintf("%s Hello on %d processors for %d elements\n",
	     doRestart?"Restarting":"Starting",CkNumPes(),nElements);
    mainProxy = thishandle;
    
    const char *dirName="arr"; 
    CProxy_Hello arr;
    if (doRestart) 
    { //Restart mode: build an empty array and fill it from the file
      arr = CProxy_Hello::ckNew();
      
      ElementRestorer restorer(dirName,arr.ckLocMgr());
      while (restorer.restore()) {}
    } else 
    { //Normal startup:
      arr = CProxy_Hello::ckNew(nElements);
      
      IndexPrinter printer("arr");
      arr.ckLocMgr()->iterate(printer);
      
      // Perform a checkpoint:
      ElementSaver saver(dirName);
      arr.ckLocMgr()->iterate(saver);
    }
	
    arr[0].SayHi(17);
  };

  void done(void)
  {
    CkPrintf("All done\n");
    CkExit();
  };
};

/*array [1D]*/
class Hello : public CBase_Hello 
{
  enum {myVal_v=0x1BADD00D};
  int myVal;
public:
  Hello()
  {
    CkPrintf("Hello %d created\n",thisIndex);
    myVal=myVal_v;
  }

  Hello(CkMigrateMessage *m) {
    CkPrintf("Hello %d migrated in\n",thisIndex);
  }
  
  void ckJustMigrated(void) {
     ArrayElement::ckJustMigrated();
     CkPrintf("Hello %d finished migrating (%08x)\n",thisIndex,myVal);
  }

  void pup(PUP::er &p) {
     ArrayElement::pup(p);
     p|myVal;
    CkPrintf("Hello %d pup'ing\n",thisIndex);
  }
  
  void SayHi(int hiNo)
  {
    CkPrintf("Hi[%d] from element %d\n",hiNo,thisIndex);
    if (thisIndex < nElements-1)
      //Pass the hello on:
      thisProxy[thisIndex+1].SayHi(hiNo+1);
    else 
      //We've been around once-- we're done.
      mainProxy.done();
  }
};

#include "hello.def.h"
