/**
 * This is a simple Threaded Charm++ program that demonstrates how to create
 * shadow arrays. The elements of the shadow array are "bound" to the TCharm
 * threads, so that the elements migrate along with the threads.
 *
 * The example has two parts, the "library" part that show how to handle creation
 * of shadow arrays and a "user" part that shows how the user code which uses
 * the library looks like.
 */

#include <charm++.h>
#include <tcharm.h>

// This starts the "library" part of the shadow creation code
#include "Shadow.decl.h"

class DistArrayShadow;

//
// this is a thread private variable that holds a pointer to the
// shadow element. A library using this use this variable to
// suspend/resume threads. In this dummy example, the shadow array is
// intended to be used by the shared memory library.
//
CtvStaticDeclare(DistArrayShadow*, __myDmaShadow);

//
// this function initialized the shadow element pointers for all the
// threads on a PE
//
void dmaNodeInit(void)
{
	CtvInitialize(DistArrayShadow*, __myDmaShadow);
}

//
// This class represents the shadow array element. It is derived from
// TCharmClient1D, a TCharm library class that helps creating shadow
// arrays for 1D arrays.
//
class DistArrayShadow : public TCharmClient1D 
{
	typedef TCharmClient1D super;

protected:

	//
	// this function is called to set the thread private variable to
	// point to this element
	// 
	virtual void setupThreadPrivate(CthThread forThread)
	{
		CtvAccessOther(forThread, __myDmaShadow) = this;
	}

	//
	// this function initializes the thread private variables on
	// during creation/migration
	//
	void initFields()
	{
		CtvAccessOther(thread->getThread(), __myDmaShadow) = this;		
	}

public:

	//
	// the shadow element constructor must call the superclass constructor
	// then tcharmClientInit() and finally set the thread to "ready to run"
	// state
	//
	DistArrayShadow(const CkArrayID &threadArrayID) : super(threadArrayID)
	{
		tcharmClientInit();
		thread->ready();
	}

	//
	// On migration, we again call migration constructor for superclass
	// and also initialize thread private variables
	//
	DistArrayShadow(CkMigrateMessage*& m) : super(m) 
	{
		thread = NULL;
		initFields();
	}

	//
	// we just migrated, so need to initialize the fields again
	//
	void ckJustMigrated()
	{
		super::ckJustMigrated();
		initFields();
	}

	void pup(PUP::er &p) 
	{
		super::pup(p); 
	}
};


//
// Each library written on top of TCharm needs to provide an Attach call
// for the library to attach to the TCharm threads. Libraries attach to 
// TCharm threads by creating the shadow arrays and defining the thread
// private variables through which they can access these shadow array
// elements
// 
void DistArrayAttach()
{
	CkArrayID threadAID;
	int nchunks;

	// the TCHARM_Attach_start() start the process of attaching a shadow
	// array to the TCharm threads. It returns an "array options" object
	// which can be used to create the shadow array.
	CkArrayOptions opts = TCHARM_Attach_start(&threadAID, &nchunks);

	// now we create another array that is the shadow of the threads array
	CProxy_DistArrayShadow shadow = CProxy_DistArrayShadow::ckNew(threadAID, opts);

	// Tell the TCharm framework that we are done with attaching the library to the
	// threads
	TCHARM_Attach_finish(shadow);
}

#include "Shadow.def.h"

// This ends the "library" part of the shadow creation code
/************************************************************************************/

// This starts the "user" part of the example

//
// This is the function that forms the "body" of the thread
//
void thread_function()
{
	ckout << "[" << CkMyPe() << "] Thread " << TCHARM_Element() << endl;
}

//
// This the user-defined function that creates threads. The user must call the Attach
// call for the library to attach to the threads
//
void TCHARM_User_setup()
{
	TCHARM_Create(TCHARM_Get_num_chunks(), thread_function);
	DistArrayAttach();
}

// this ends the "user" part of the code

