package projections.analysis;

/*
LogReader.java: Charm++ projections.  
Converted to Java by Theckla Louchios
Improved by Orion Sky Lawlor, olawlor@acm.org

Reads .log files generated by a charm++ program compiled with
"-tracemode projections".  Converts .log files into 
gigantic multidimentional arrays indexed by entry point,
processor, and time interval.
*/
import java.lang.*;
import java.io.*;
import java.util.*;

import javax.swing.*;

import projections.misc.*;
import projections.gui.*;

public class LogReader 
    extends ProjDefs
{
    //sysUsgData[SYS_Q] is length of message queue
    public static final int SYS_Q=0; 
    //sysUsgData[SYS_CPU] is percent processing time
    public static final int SYS_CPU=1; 
    //sysUsgData[SYS_IDLE] is percent idle time
    public static final int SYS_IDLE=2; 
    //Magic entry method (indicates "idle")
    public static final int IDLE_ENTRY=-1;
    //Number of creations bin 
    public static final int CREATE=0; 
    //Number of invocations bin
    public static final int PROCESS=1; 
    //Time (us) spent processing bin
    public static final int TIME=2; 

    private int[][][] sysUsgData;
    private int[][][][] userEntries;
    private int[][][][] categorized;
    private int numProcessors;
    private int numUserEntries; //Number of user entry points
    private long startTime; //Time the current entry method started
    private int currentEntry; //Currently executing entry method
    private int currentMtype; //Current message type
    private int interval;//Current interval number
    private int curPe;//Current source processor #
    private int numIntervals;//Total number of intervals
    private long intervalSize;//Length of an interval (us)
    private int intervalStart;// start of range of interesting intervals
    private int intervalEnd; // end of range of interesting intervals

    private int processing;
    private boolean byEntryPoint;
    
    // **CW** variables to support delta encoding
    private long prevTime = 0;
    private boolean deltaEncoded = false;
    private int tokenExpected = 2;

    public long getIntervalSize() {
	return intervalSize;
    }
	
    /**
       Add the given amount of time to the current entry for interval j
    */
    final private void addToInterval(int extra,int j,boolean maxPercents)
    {
	if (processing<=0) {
	    return; //Not processing at all
	}
	if (j < intervalStart || j > intervalEnd) {  // Not within interest
	    return;
	}
	if (currentEntry == IDLE_ENTRY) { //Idle time
	    sysUsgData[SYS_IDLE][curPe][j-intervalStart] += 
		maxPercents?100:extra;
	} else { //Processing an entry method
	    sysUsgData[SYS_CPU][curPe][j-intervalStart] += 
		maxPercents?100:extra;
	    if (byEntryPoint) {
		userEntries[currentEntry][TIME][curPe][j-intervalStart] += 
		    extra; 
		int catIdx = mtypeToCategoryIdx(currentMtype); 
		if (catIdx!=-1) {
		    categorized[catIdx][TIME][curPe][j-intervalStart]+=extra;
		}
	    }
	}
	startTime+=extra;
    }

    /**
       Add this entry point to the counts
    */
    final private void count(int mtype,int entry,int TYPE)
    {
	if (!byEntryPoint) { 
	    return;
	}
	if (userEntries[entry][TYPE][curPe]==null) {
	    userEntries[entry][TYPE][curPe]=new int[numIntervals+1];
	}
	if (TYPE==PROCESS && 
	    userEntries[entry][TIME][curPe] == null) { //Add a time array, too
	    userEntries[entry][TIME][curPe]=new int[numIntervals+1];
	}
	userEntries[entry][TYPE][curPe][interval-intervalStart]++;
	
	int catIdx=mtypeToCategoryIdx(mtype);
	if (catIdx!=-1) {
	    if (categorized[catIdx][TYPE][curPe]==null) {
		categorized[catIdx][TYPE][curPe]=new int[numIntervals+1];
		if (TYPE==PROCESS) { //Add a time array, too
		    categorized[catIdx][TIME][curPe]=new int[numIntervals+1];
		}
	    }
	    categorized[catIdx][TYPE][curPe][interval-intervalStart]++;
	}
    }

    /**
       This is called when a log entry crosses a timing interval boundary
    */
    private void fillToInterval(int newinterval)
    {
	if (interval>=newinterval) {
	    return; //Nothing to do in this case
	}
	
	//Finish off the current interval
	int extra = (int) ((interval+1)*intervalSize - startTime);
	addToInterval(extra,interval,false);

	//Convert system usage and idle time from us to percent of an interval
	rescale(interval);

	//Fill in any intervals we would skip over completely
	for (int j=interval+1; j<newinterval; j++) {
	    addToInterval((int)intervalSize,j,true);
	}

	interval = newinterval;
    }

    public int[][][][] getSystemMsgs() { 
	return categorized; 
    }

    public int[][][] getSystemUsageData() { 
	return sysUsgData; 
    }

    public int[][][][] getUserEntries() {
	return userEntries;
    }
    
    // the interval values used are absolute. Only when array access is
    // required would it be offset by intervalStart.
    private void intervalCalc(int type, int mtype, int entry, long time) 
	throws IOException 
    {
	fillToInterval((int)(time/intervalSize));
	switch (type) {
	case ENQUEUE:
	    if (interval >= intervalStart && interval <= intervalEnd) {
		// Lengthen system queue
		sysUsgData[SYS_Q][curPe][interval-intervalStart]++;
	    }
	    break;
	case CREATION:
	    if (interval >= intervalStart && interval <= intervalEnd) {
		// Lengthen system queue
		sysUsgData[SYS_Q][curPe][interval-intervalStart]++;
		count(mtype,entry,CREATE);
	    }
	    break;
	case BEGIN_PROCESSING:
	    processing++;
	    startTime = time;
	    if (interval >= intervalStart && interval <= intervalEnd) {
		//Shorten system queue
		sysUsgData[SYS_Q][curPe][interval-intervalStart]--;
		count(currentMtype = mtype,currentEntry = entry,PROCESS);
	    }
	    break;
	case END_PROCESSING:
	    if (interval >= intervalStart && interval <= intervalEnd) {
		addToInterval((int)(time-startTime),interval,false);
	    }
	    processing--;
	    break;
	case BEGIN_IDLE:
	    processing++;
	    startTime = time;
	    currentEntry = IDLE_ENTRY;
	    break;
	case END_IDLE:
	    if (interval >= intervalStart && interval <= intervalEnd) {
		addToInterval((int)(time-startTime),interval,false);
	    }
	    processing--;
	    break;
	default:
	    System.out.println("Unhandled type "+type+" in logreader!");
	    break;
	}
    }

    /**
       Maps a message type to a categorized system message number
    */
    final private int mtypeToCategoryIdx(int mtype) {
	switch (mtype) {
	case NEW_CHARE_MSG: 
	    return 0;
	case FOR_CHARE_MSG: 
	    return 1;
	case BOC_INIT_MSG: 
	    return 2;
	case LDB_MSG: 
	    return 3;
	case QD_BROADCAST_BOC_MSG: case QD_BOC_MSG:
	    return 4;
	default:
	    return -1;
	}
    }

    /**
       read log file, with one more parameter containing 
       a list of processors to read.
    */
    public void read(long reqIntervalSize,
		     int NintervalStart, int NintervalEnd,
		     boolean NbyEntryPoint, OrderedIntList processorList)
    {
	int type;
	int mtype;
	long time;
	int entry;
	int event;
	int pe;
	int msglen;
	
	String logHeader;

	numProcessors = Analysis.getNumProcessors();
	numUserEntries = Analysis.getNumUserEntries();
	intervalSize = reqIntervalSize;
	intervalStart = NintervalStart;
	intervalEnd = NintervalEnd;
	numIntervals = intervalEnd - intervalStart + 1;
	byEntryPoint=NbyEntryPoint;
	
	// assume full range of processor if null
        if (processorList == null) {
	    processorList = new OrderedIntList();
	    for (pe=0; pe<numProcessors; pe++) {
		processorList.insert(pe);
	    }
        } else {
	    // **CW** required to set numProcessors to the correct values.
	    numProcessors = processorList.size();
	}

	ProgressMonitor progressBar = 
	    new ProgressMonitor(Analysis.guiRoot, "Reading log files",
				"", 0, numProcessors);
	progressBar.setNote("Allocating Global Memory");
	progressBar.setProgress(0);
	sysUsgData = new int[3][numProcessors][];
	if (byEntryPoint) {
	    userEntries = new 
		int[numUserEntries][3][numProcessors][numIntervals];
	    categorized = new int[5][3][numProcessors][];
	}
	int seq = 0;
        processorList.reset();
	int nPe=processorList.size();
 	curPe = processorList.nextElement();
	for (;curPe!=-1; curPe=processorList.nextElement())
	    try {
		progressBar.setProgress(seq);
		progressBar.setNote("Allocating Memory for PE " + curPe);
		// gzheng: allocate sysUsgData only when needed.
                sysUsgData[0][curPe] = new int [numIntervals+1];
                sysUsgData[1][curPe] = new int [numIntervals+1];
                sysUsgData[2][curPe] = new int [numIntervals+1];
		progressBar.setNote("Reading data for PE " + curPe);
		if (progressBar.isCanceled()) {
		    // clear all data and return
		    userEntries = null;
		    categorized = null;
		    sysUsgData = null;
		    return;
		}
		seq ++;
		processing = 0;
		interval = 0;
	        currentEntry = -1;
	        startTime =0;
		FileReader file = new FileReader(Analysis.getLogName(curPe));
		AsciiIntegerReader log = 
		    new AsciiIntegerReader(new BufferedReader(file));

		// **CW** first line is no longer junk.
		// With the advent of the delta-encoding format, it should
		// contain an additional field which specifies if the log file
		// is a delta-encoded file.
		logHeader = log.readLine();
		StringTokenizer headerTokenizer = 
		    new StringTokenizer(logHeader);
		// **CW** a hack to avoid parsing the string - simply count
		// the number of tokens.
		if (Analysis.getVersion() >= 6.0)  tokenExpected = 3;
		if (headerTokenizer.countTokens() > tokenExpected) {
		    deltaEncoded = true;
		} else {
		    deltaEncoded = false;
		}
		log.nextLine(); // clear the rest of the first line.

		// The Begin Computation line (for now) is useless unless
		// delta-encoding (which also seems to be useless) is
		// employed.
		if (deltaEncoded) {
		    log.nextInt();
		    prevTime = log.nextLong();
		} else {
		    log.nextLine(); // ignore second line.
		}

		int nLines=2;

		boolean isProcessing = false;
		try { 
		    while (true) { //EOFException will terminate loop
			log.nextLine();//Skip any junk from previous line
			type = log.nextInt();
		  	nLines++;
			switch (type) {
			case BEGIN_IDLE: case END_IDLE:
			    if (deltaEncoded) {
				prevTime += log.nextLong();
				time = prevTime;
			    } else {
				time = log.nextLong();
			    }
			    pe = log.nextInt();
			    intervalCalc(type, 0, 0, time);
			    break;
			case CREATION:
			    mtype = log.nextInt();
			    entry = log.nextInt();
			    if (deltaEncoded) {
				prevTime += log.nextLong();
				time = prevTime;
			    } else {
				time = log.nextLong();
			    }
			    event = log.nextInt();
			    pe = log.nextInt();
			    if (Analysis.getVersion() > 1.0) {
				msglen = log.nextInt();
			    } else {
				msglen = -1;
			    }
			    intervalCalc(type, mtype, entry, 
					 time);
			    break;
			case BEGIN_PROCESSING: 
			    if (isProcessing) {
				// bad, ignore.
				break;
			    }
			    mtype = log.nextInt();
			    entry = log.nextInt();
			    if (deltaEncoded) {
				prevTime += log.nextLong();
				time = prevTime;
			    } else {
				time = log.nextLong();
			    }
			    event = log.nextInt();
			    pe = log.nextInt();
			    if (Analysis.getVersion() > 1.0) {
				msglen = log.nextInt();
			    } else {
				msglen = -1;
			    }
			    intervalCalc(type, mtype, entry, 
					 time);
			    isProcessing = true;
			    break;
                        case END_PROCESSING:
			    if (!isProcessing) {
				// bad, ignore.
				break;
			    }
			    mtype = log.nextInt();
			    entry = log.nextInt();
			    if (deltaEncoded) {
				prevTime += log.nextLong();
				time = prevTime;
			    } else {
				time = log.nextLong();
			    }
			    event = log.nextInt();
			    pe = log.nextInt();
			    if (Analysis.getVersion() > 1.0) {
				msglen = log.nextInt();
			    } else {
				msglen = -1;
			    }
			    intervalCalc(type, mtype, entry, 
					 time);
			    isProcessing = false;
			    break;
			case ENQUEUE:
			    mtype = log.nextInt();
			    if (deltaEncoded) {
				prevTime += log.nextLong();
				time = prevTime;
			    } else {
				time = log.nextLong();
			    }
			    event = log.nextInt();
			    pe = log.nextInt();
			    intervalCalc(type, mtype, 0, time);
			    break;
			case END_COMPUTATION:
			    // end computation uses absolute timestamps.
			    time = log.nextLong();
			    fillToInterval(numIntervals);
			    break;
			case USER_EVENT:
			case USER_EVENT_PAIR:
			    // get time stamp and ignore the rest
			    if (deltaEncoded) {
				// first field is user message id
				log.nextInt();
				prevTime += log.nextLong();
			    }
			    break;
			case DEQUEUE:
			    // get time stamp and ignore the rest
			    if (deltaEncoded) {
				// first field is message type
				// this case is separated from USER*
				// for readability.
				log.nextInt();
				prevTime += log.nextLong();
			    }
			    break;
			case BEGIN_PACK:
			case END_PACK:
			case BEGIN_UNPACK:
			case END_UNPACK:
			case BEGIN_INTERRUPT:
			case END_INTERRUPT:
			    // get time stamp and ignore the rest
			    if (deltaEncoded) {
				prevTime += log.nextLong();
			    }
			    break;
			case INSERT:
			case FIND:
			case DELETE:
			    // included for completeness
			    // not even generated by the tracing code anymore.
			    break; // just ignore.
 			default:
			    // **CW** For delta encoding, we cannot simply
			    // ignore unknown events. All events have to
			    // be processed (at least up to the timestamp)
			    // so that the timestamps can add up right.
			    System.out.println("Warning: Unknown Event " + 
					       type +
					       "! " +
					       "May mess delta encoding!");
			    break;//Just skip this line
 			}
		    }
		} catch (EOFException e) {
		    log.close();
		} catch (IOException e) {
		    log.close();
		}
	    } catch (IOException e) { 
		System.out.println("Exception reading log file #"+curPe); 
		return;
	    }
	progressBar.close();
    }

    //Convert system usage and idle time from us to percent of an interval
    private void rescale(int j) {
	if (j < intervalStart || j > intervalEnd) {
	    return;
	}
	sysUsgData[SYS_CPU ][curPe][j-intervalStart] =
	    (int)(sysUsgData[SYS_CPU ][curPe][j-intervalStart]*100/intervalSize);
	sysUsgData[SYS_IDLE][curPe][j-intervalStart] =
	    (int)(sysUsgData[SYS_IDLE][curPe][j-intervalStart]*100/intervalSize);
    }

}
