/*
Main routine for grid split-- read the input 
mesh file, write out the .mblock files.

Orion Sky Lawlor, olawlor@acm.org, 6/7/2001
*/
#include <stdio.h>
#include "makeflo.h"

void checkError(const char *errCode) {
  if (errCode==NULL) return; //No problem
  fprintf(stderr,"FATAL ERROR! %s\n",errCode);
  exit(1);
}

void printUsage(const char *why)
{
	printf("Usage: makemblock \n"
	  "     <.grd, or .msh mesh; and .inp and .bc file> \n"
	  "     <# output chunks> <output file prefix>\n"
	  "  Makemblock reads a mesh (first parameter) and boundary condition\n"
	  "(.inp) list, partitions the mesh into the requested number of\n"
	  "chunks, and writes out mblock input files.\n"
	  "  The mesh formats supported are:\n"
	  "<*.grd>  A Gridgen ASCII double-precision mesh description\n"
	  "<*.msh>  A double-precision binary Rocflo mesh file\n"
	  "<*.hdf>  A double-precision 3D HDF mesh file\n"
	  "<*.mblk> A double-precision 3D Mblock mesh file\n"
	  "  When the mesh consists of multiple files, give only the name of\n"
	  "the first file (e.g., 'tstflo_001.hdf').  The number of blocks and\n"
	  "numeric format will be automatically determined.\n"
	  "Part of the Charm++ Tools.  Version " MAKEFLO_VERSION "\n"
	); 
	if (why!=NULL)
		printf("Exiting> %s\n",why);
	exit(1);
}

makefloParam parameters;

int main(int argc,char *argv[]) 
{
//Parse the command line arguments
	if (argc<4) printUsage("Not enough command-line arguments");
	int curArg=1;

	while (argv[curArg][0]=='-') {
		if (0==strcmp(argv[curArg],"-2D")) {
			parameters.skipAxis=2; //Don't bother about the z axis.
			curArg++;
		}
		else if (0==strcmp(argv[curArg],"-top")) {
			parameters.topologyOnly=1; //Only write .bblk
			curArg++;
		}
		else
			printUsage("Unrecongized parameter");
	}

	if (curArg+3!=argc) printUsage("Too many arguments");
	const char *inMesh=argv[curArg+0];
	string inpFile=replaceExtention(argv[curArg+0],".inp");
	int nPieces=atoi(argv[curArg+1]);
	const char *outMesh=argv[curArg+2];
	
	vector<block *> blocks;
	
//Read the blocks of the mesh
	{ //<- scoping block so r's destructor gets called
		blockReader r(blocks);
		checkError(r.read(inMesh));
		printf("Mesh file read successfully.\n");
	}
	
//Read the boundary conditions
	checkError(readBoundaries(blocks,inpFile.c_str()));
	printf("Boundary conditions read successfully.\n");
	
//Split the blocks (for greater parallelism)
	checkError(splitBlocks(blocks,nPieces));
	printf("Split blocks for %d PEs (%d blocks generated).\n",nPieces,blocks.size());	

//Build the block faces & associate all shared nodes
	buildFaces(blocks,false);
	printf("Nodes matched successfully.\n");
	
//Write out the blocks
	checkError(writeMblock(blocks,outMesh));
	printf("Block files written successfully\n");
	return 0;
}






