/*
Simple Array Reduction test program--
Orion Sky Lawlor, olawlor@acm.org, 11/15/1999

This program shows how to use the Array mangager's 
reduction routines.
 */
#include <stdio.h>
#include <string.h>
#include "hello.decl.h"

CProxy_main mid;
CProxy_Hello arr;
int nElements;

class redData {
	int sourceIndex;
	int a;
	double b;
public:
	redData(int fromIndex) {
		sourceIndex=fromIndex;
		a=fromIndex*10;
		b=fromIndex*0.1;
	}
	redData(void) {
		sourceIndex=-1;
		a=0;
		b=0.0;
	}
	
	void accumulate(const redData &r) {
		a+=r.a;
		b+=r.b;
	}
	void print(void) {
		CkPrintf("%s result: a=%d, b=%f\n",
		       sourceIndex==-1?"composite":"single",
		       a,b);
	}
};

//A custom reduction function, for reducing my special data type
CkReductionMsg *customReductionFn(int nMsg,CkReductionMsg **msgs)
{
	redData ret;
	for (int i=0;i<nMsg;i++) {
		if (msgs[i]->getSize()!=sizeof(redData)) {
			CkError("Unexpected message size %d!\n",msgs[i]->getSize());
			CkAbort("Unexpected data passed to custom reduction fn!\n");
		}
		const redData *d=(const redData *)(msgs[i]->getData());
		ret.accumulate(*d);
	}
	return CkReductionMsg::buildNew(sizeof(ret),&ret);
}

//We have to register the custom reduction function on each node
CkReduction::reducerType customReductionFnIndex;
void initRednTest(void)
{
	//Register my reduction function
	customReductionFnIndex=CkReduction::addReducer(customReductionFn);
}


typedef struct {
	int reductionsComplete; //Number of finished reductions
	int reductionsTotal; //Number of reductions to do
} myReductionCounter;

void printReductionHandler(void *param,int dataSize,void *data)
{
	myReductionCounter *c=(myReductionCounter *)param;
	CkPrintf("**********************************************\n");
	CkPrintf("*** Final reduction handler called-- %d bytes:\n",dataSize);
	switch(c->reductionsComplete) {
	case 0: {//Just integers
		int *id=(int *)data;
		CkPrintf("*** data={%d,%d,%d}\n",id[0],id[1],id[2]);
		if (id[0]<nElements)
			CkAbort("Not every element contributed!");
		else if (id[0]>nElements)
			CkAbort("Some elements contributed more than once!");
	  }
	  break;
	case 1:
          {   //Is a reduction set--parse each element
		CkReduction::setElement *cur=(CkReduction::setElement *)data;
		while (cur!=NULL)
		{
			CkPrintf("*** %d bytes: '%s'\n",cur->dataSize,(char *)cur->data);
			cur=cur->next();
		}
	  }
	  break;
	case 2: //Custom reduction type and function
		((redData *)data)->print();
		break;
	default:
		CkAbort("Too many reductions reported results!\n");
	};
	
	CkPrintf("**********************************************\n");
	c->reductionsComplete++;
	if (c->reductionsComplete>=c->reductionsTotal)
	{//We're finished-- tell main to quit
		mid.maindone();
	} else 
	{//Broadcast a "Hi" again
		 //for (int i=0;i<nElements;i++) arr[i].DoIt();
		 arr.DoIt();
	}
}

class main : public Chare
{
public:
  main(CkMigrateMessage *m) {}
  main(CkArgMsg* m)
  {
    nElements=5;
    if(m->argc > 1) nElements = atoi(m->argv[1]);
    delete m;
    CkPrintf("Running Hello on %d processors for %d elements\n",
	     CkNumPes(),nElements);
    mid = thishandle;
    arr = CProxy_Hello::ckNew(nElements);

    myReductionCounter *c=new myReductionCounter;
    c->reductionsComplete=0;
    c->reductionsTotal=3;

    arr.setReductionClient(printReductionHandler,(void *)c);
    
    arr.DoIt();
  };

  void maindone(void)
  {
    CkPrintf("All done\n");
    CkExit();
  };
};

class Hello : public ArrayElement1D
{
	 int nTimes;
public:
  Hello() {
	  CkPrintf("Creating element %d on PE %d\n",thisIndex,CkMyPe());
	  nTimes=0;
  }

  Hello(CkMigrateMessage *m) {}

  void DoIt(void)
  {
	  CkPrintf("Contributing to reduction %d, element %04d\n",nTimes,thisIndex);
	  if (nTimes==0) 
	  {//Contribute integers
		  int f[3];
		  f[0]=1;
		  f[1]=thisIndex*(thisIndex+1)+2;
		  f[2]=thisIndex*(thisIndex+4)-2;
		  contribute(3*sizeof(int),(void *)f,CkReduction::sum_int);
	  }
	  else if (nTimes==1)
	  { //Reducing to a set-- contribute some random string
		  char buf[100];
		  sprintf(buf,"I'm %d, running on PE %d.",thisIndex,CkMyPe());
		  contribute(strlen(buf)+1,(void *)buf,CkReduction::set);
	  }
	  else if (nTimes==2)
	  { //Custom reduction function
		  redData r(thisIndex);
		  contribute(sizeof(r),&r,customReductionFnIndex);
	  }
	  else
		  CkAbort("Too many calls to DoIt!\n");
	  nTimes++;
  }
};

#include "hello.def.h"

