/*
A tiny FEM mesh examination and reassembly tool:

Run this program like
	./pgm +vp60 -read
to read in fem_mesh_vp60_* files, reassemble them,
convert to .faces/.points and .noboite, and present
them for NetFEM.  Because the program waits forever
for a NetFEM client, you'll have to press control-C to 
kill it off.

  Orion Sky Lawlor, olawlor@acm.org, 2003/5/8
*/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <vector>
#include "charm++.h"
#include "fem.h"
#include "netfem.h"
#include "tcharmc.h"
#include "ckvector3d.h"

extern "C" void mesh_updated(int param);


extern "C" void
init(void)
{
  FEM_Print("Usage: copy fem_vp files to this directory and run with '-read'");
  FEM_Done();
}


/** Print this mesh's constituents: */
void printFEM(int mesh) {
	char storage[100], storage2[100];
	int entities[1000];
	int nEntities=FEM_Mesh_get_entities(mesh,entities);
	for (int e=0;e<nEntities;e++) {
		int entity=entities[e];
		int len=FEM_Mesh_get_length(mesh,entity);
		printf("Entity %d: %s, number=%d\n",
			e,FEM_Get_entity_name(entity,storage),len);
		int attrs[1000];
		int nAttr=FEM_Mesh_get_attributes(mesh,entity,attrs);
		for (int a=0;a<nAttr;a++) {
			int attr=attrs[a];
			int wid=FEM_Mesh_get_width(mesh,entity,attr);
			int dt=FEM_Mesh_get_datatype(mesh,entity,attr);
			printf("   Attribute %d: %s, width=%d, datatype=%s\n",
				a,FEM_Get_attr_name(attr,storage), wid,
				FEM_Get_datatype_name(dt,storage2));
		}
	}
}

/// Return the size, in bytes, of this FEM type
int typeSize(int datatype) {
	switch (datatype) {
	case FEM_BYTE: return 1;
	case FEM_INT: return sizeof(int);
	case FEM_FLOAT: return sizeof(float);
	case FEM_DOUBLE: return sizeof(double);
	case FEM_INDEX_0: return sizeof(int);
	case FEM_INDEX_1: return sizeof(int);
	}
}

/** Copy this mesh's data into this other mesh.
    This is needed because FEM's reading and reassembly
    use different (implicit) meshes, so for now there 
    has to be a copy from one to the other.
 */
void copyFEM(int srcMesh,int destMesh) {
	int entities[1000];
	int nEntities=FEM_Mesh_get_entities(srcMesh,entities);
	for (int e=0;e<nEntities;e++) {
		int entity=entities[e];
		int len=FEM_Mesh_get_length(srcMesh,entity);
		int attrs[1000];
		int nAttr=FEM_Mesh_get_attributes(srcMesh,entity,attrs);
		for (int a=0;a<nAttr;a++) {
			int attr=attrs[a];
			if (attr==FEM_SPARSE_ELEM) 
			{ /* can't copy these until they get renumbered properly from global numbers... */
				continue;
			}
			int wid=FEM_Mesh_get_width(srcMesh,entity,attr);
			int dt=FEM_Mesh_get_datatype(srcMesh,entity,attr);
			if (attr==FEM_CONN || attr==FEM_GLOBALNO || attr==FEM_SPARSE_ELEM)
				dt=FEM_INDEX_0; /* these guys use index data... */
			char *data=new char[len*wid*typeSize(dt)];
			FEM_Mesh_data(srcMesh,entity,attr,
				data, 0,len, dt,wid);
			FEM_Mesh_data(destMesh,entity,attr,
				data, 0,len, dt,wid);
			delete[] data;
		}
	}
}

/** A BoundaryDest accepts incoming boundary triangles.
*/
class BoundaryDest {
public:
	/**
	Add the boundary condition "bc" along the triangle formed
	by these three nodes.
	*/
	virtual void addTriangle(int bc,const int *globalNodes) =0;
};

/** Extract the (Andreas-specific) triangular boundary faces
    and boundary type flags into this class.
*/
void extractBoundary(int srcMesh,BoundaryDest &dest) {
	int entities[1000];
	int nEntities=FEM_Mesh_get_entities(srcMesh,entities);
	for (int e=0;e<nEntities;e++) {
		int entity=entities[e];
		if (entity<FEM_SPARSE) continue; //Only want boundaries
		int len=FEM_Mesh_get_length(srcMesh,entity);
		if (len==0) continue; // Skip over missing boundaries
		int wid=FEM_Mesh_get_width(srcMesh,entity,FEM_CONN);
		if (wid!=3) continue; // Skip over boundary dots and quads (FIXME!)
		int *conn=new int[wid*len];
		FEM_Mesh_data(srcMesh,entity,FEM_CONN,
				conn, 0,len, FEM_INDEX_0,wid);
		int *type=new int[len];
		FEM_Mesh_data(srcMesh,entity,FEM_DATA,
				type, 0,len, FEM_INT,1);
		for (int i=0;i<len;i++) {
			dest.addTriangle(type[i],&conn[wid*i]);
		}
		delete[] type;
		delete[] conn;
	}
}


/// Create a NetFEM-compatible list from the incoming 
/// boundary faces:
class ListBoundaryDest : public BoundaryDest {
public:
	std::vector<int> &conn;
	std::vector<double> &type;
	ListBoundaryDest(std::vector<int> &c_,std::vector<double> &t_)
		:conn(c_), type(t_) {}
	void addTriangle(int bc,const int *globalNodes) {
		type.push_back(bc);
		for (int i=0;i<3;i++)
			conn.push_back(globalNodes[i]);
	}
};


/** Upload this (Andreas-specific) tet mesh and boundary to NetFEM 
  FIXME: do this for real... and don't leak memory via FEM_POINTAT
*/
void toNetFEM(int partition, int srcMesh) {
	NetFEM n=NetFEM_Begin(partition,0,3,NetFEM_POINTAT);
	
// Nodes:
	int nNodes=FEM_Mesh_get_length(srcMesh,FEM_NODE);
	double *nodeLoc=new double[3*nNodes];
	FEM_Mesh_data(srcMesh,FEM_NODE,FEM_DATA+0,
				nodeLoc, 0,nNodes, FEM_DOUBLE,3);
	NetFEM_Nodes(n,nNodes,(double *)nodeLoc,"Position (m?)");
	CkPrintf("Uploaded %d nodes\n",nNodes);

// Tets: (FIXME! Add other element types! )
if (0) {
	int tetEnt=FEM_ELEM+1;
	int nTet=FEM_Mesh_get_length(srcMesh,tetEnt);
	{
	  int wid=4;
	  int *tetConn=new int[wid*nTet];
	  FEM_Mesh_data(srcMesh,tetEnt,FEM_CONN,
				tetConn, 0,nTet, FEM_INDEX_0,wid);
	  NetFEM_Elements(n,nTet,wid,(int *)tetConn,"Tets");
	}
	if (1) {
	  int wid=FEM_Mesh_get_width(srcMesh,tetEnt,FEM_DATA+0);
	  double *tetData=new double[wid*nTet];
	  FEM_Mesh_data(srcMesh,tetEnt,FEM_DATA+0,
				tetData, 0,nTet, FEM_DOUBLE,wid);
	  NetFEM_Scalar(n,tetData,5,"Solution data (varies)");
	}
	CkPrintf("Uploaded %d tets\n",nTet);
}

if (1) {
// Boundaries:
	std::vector<int> *boundConn=new std::vector<int>;
	std::vector<double> *boundType=new std::vector<double>;
	ListBoundaryDest dest(*boundConn,*boundType);
	extractBoundary(srcMesh,dest);
	int nBound=boundType->size();
	
	CkPrintf("Uploaded %d boundary faces\n",nBound);
	NetFEM_Elements(n,nBound,3,&dest.conn[0],"Boundary Faces");
	if (1)
		NetFEM_Scalar(n,&dest.type[0],1,"Boundary Types");
}
	
	NetFEM_End(n);
	
}


/// Renumber these incoming boundary faces to have local node numbers.
class RenumberBoundaryDest : public BoundaryDest {
	const CkVector3d *globalNodes; //Location of global nodes
	int *localFmGlobal; // Maps a global node number to (-1) or a local node number.
public:
	std::vector<CkVector3d> localNodes; // Locations of local nodes
	std::vector<int> triBC; // Boundary condition flags for each triangle
	std::vector<int> triConn; // Local node connectivity for each triangle
	
	RenumberBoundaryDest(const CkVector3d *globalNodes_,int nGlobalNodes) 
		:globalNodes(globalNodes_)
	{
		localFmGlobal=new int[nGlobalNodes];
		for (int i=0;i<nGlobalNodes;i++) localFmGlobal[i]=-1;
	}
	~RenumberBoundaryDest() {
		delete[] localFmGlobal;
	}
	
	/// Return this global node's local index, creating if needed.
	int lookupNode(int g) {
		int l=localFmGlobal[g];
		if (l==-1) { // A new local node: add it
			l=localNodes.size();
			localFmGlobal[g]=l;
			localNodes.push_back(globalNodes[g]);
		}
		return l;
	}
	
	void addTriangle(int bc,const int *conn) {
		triBC.push_back(bc);
		for (int i=0;i<3;i++) {
			triConn.push_back(lookupNode(conn[i]));
		}
	}
};

/** Write the boundary of this mesh to these YAMS .msh2 files:
*/
void writeYAMS(int srcMesh,const char *pointsName,const char *facesName)
{
	int nNodes=FEM_Mesh_get_length(srcMesh,FEM_NODE);
	CkVector3d *globalNodes=new CkVector3d[nNodes];
	FEM_Mesh_data(srcMesh,FEM_NODE,FEM_DATA+0, 
		globalNodes, 0,nNodes, FEM_DOUBLE, 3);
	RenumberBoundaryDest b(globalNodes,nNodes);
	extractBoundary(srcMesh,b);
	delete[] globalNodes;
	
	FILE *points=fopen(pointsName,"w");
	if (points==NULL) {CkError("Can't create output file '%s'\n",pointsName);return;}
	int nPoints=b.localNodes.size();
	fprintf(points,"%d\n",nPoints);
	for (int p=0;p<nPoints;p++) {
		CkVector3d v=b.localNodes[p];
		int ref=0;
		fprintf(points,"%f %f %f %d\n",v.x,v.y,v.z,ref);
	}
	fclose(points); 
	
	FILE *faces=fopen(facesName,"w");
	if (points==NULL) {CkError("Can't create output file '%s'\n",facesName);return;}
	int nFaces=b.triBC.size();
	fprintf(faces,"%d\n",nFaces);
	for (int f=0;f<nFaces;f++) {
		int edgeRef=0;
		int nPoints=3;
		const int *conn=&b.triConn[nPoints*f];
		fprintf(faces,"%d  %d %d %d  %d  %d %d %d\n",
			nPoints, conn[0]+1,conn[1]+1,conn[2]+1, b.triBC[f], 
			edgeRef,edgeRef,edgeRef);
	}
	fclose(faces);
	CkPrintf("Wrote YAMS output to %s/%s\n",pointsName,facesName);
}

/** Write the interior tets of this mesh to this TetMesh .noboite file.
*/
void writeNOBOITE(int srcMesh,const char *noboiteName) {
	FILE *f=fopen(noboiteName,"w");
	if (f==NULL) {CkError("Can't create output file '%s'\n",noboiteName);return;}
	
	int nNodes=FEM_Mesh_get_length(srcMesh,FEM_NODE);
	CkVector3d *nodes=new CkVector3d[nNodes];
	FEM_Mesh_data(srcMesh,FEM_NODE,FEM_DATA+0, 
		nodes, 0,nNodes, FEM_DOUBLE, 3);
	
	int nTets=FEM_Mesh_get_length(srcMesh,FEM_ELEM+1);
	int nodePerTet=4;
	int *conn=new int[nodePerTet*nTets];
	FEM_Mesh_data(srcMesh,FEM_ELEM+1,FEM_CONN, 
		conn, 0,nTets, FEM_INDEX_0, nodePerTet);
	
	fprintf(f,"%d %d -1  -1 -1 -1\n"
		"-1 -1 -1  -1 -1 -1\n"
		"-1 -1 -1  -1 -1\n", nTets, nNodes);
	
	for (int t=0;t<nTets;t++) {
		const int *c=&conn[nodePerTet*t];
		fprintf(f,"%d %d %d %d\n",c[0]+1,c[1]+1,c[2]+1,c[3]+1);
	}
	
	for (int n=0;n<nNodes;n++) {
		CkVector3d v=nodes[n];
		fprintf(f,"%f %f %f\n",v.x,v.y,v.z);
	}
	
	delete[] conn;
	delete[] nodes;
	fclose(f);
}


extern "C" void
driver(void)
{
  int myID=FEM_My_partition();
  int srcMesh=FEM_Mesh_default_read();
  if (myID==0) {
	FEM_Print("----------- begin driver ------------");
        printFEM(srcMesh);
  }
  if (CkNumPes()>1) { // HACK: on multiple PE's, show unassembled mesh
  	toNetFEM(myID,srcMesh);
  }
  double start=CkWallTimer();
  copyFEM(srcMesh,FEM_Mesh_default_write());
  if (myID==0) CkPrintf("Copying mesh took %.3f s\n",CkWallTimer()-start); 
  start=CkWallTimer();
  FEM_Update_mesh(mesh_updated,123,FEM_MESH_FINALIZE);
  if (myID==0) CkPrintf("Reassembling mesh took %.3f s\n",CkWallTimer()-start); 

// Now hang until requests come in:
  if (myID==0) FEM_Print("----------- Waiting for NetFEM client ------------");
  while (1) {
	TCHARM_Yield();
	sleep(1);
  }
  
  if (myID==0) FEM_Print("----------- end driver ------------");
}

extern "C" void
mesh_updated(int param)
{
  CkPrintf("mesh_updated(%d) called:\n",param);
  int srcMesh=FEM_Mesh_default_read();
  printFEM(srcMesh);
  writeYAMS(srcMesh,"out.points","out.faces");
  writeNOBOITE(srcMesh,"out.noboite");
  if (CkNumPes()==1) { // HACK: on one PE, show *assembled* mesh
  	toNetFEM(0,srcMesh);
  }
}
