/* File			: maxReduceNV.C
 * Description	: Sample application for Blue Gene emulator (converse version)
 *				  Computing Maximum of numbers stored at each node. 
 * Author		: Arun Singla
 * Version		: 1.0; March 26, 2001
 *
 * Note			: The same program can be written much more efficiently. This program 
 *				  explains how to use several handlers, how to use node private data, 
 *				  and how to reduce result of computations at each node to one node.
 */

#include <stdlib.h>
#include "blue.h"

#define A_SIZE 4

/* Handler function Identifiers */
int reduceID;
int computeMaxID;
int contributeID;

/* Handler function prototypes */
extern "C" void reduce    (char *);
extern "C" void computeMax(char *);
extern "C" void contribute(char *);

class computeMaxMsg
{
public:
  char core[CmiBlueGeneMsgHeaderSizeBytes];
  int max;
  void *operator new(size_t s) { return CmiAlloc(s); }
  void operator delete(void* ptr) { CmiFree(ptr); }
};

/*
typedef struct userDataStruct
{
  int data[A_SIZE] ;
  int count ;
} userData ;
*/

typedef int nodeData[A_SIZE];
BnvDeclare(nodeData, data);
BnvDeclare(int, count);

void BgEmulatorInit(int argc, char **argv) 
{
  if (argc < 6) { 
    CmiPrintf("Usage: <program> <x> <y> <z> <numCommTh> <numWorkTh>\n"); 
    BgShutdown();
  }

  /* set machine configuration */
  BgSetSize(atoi(argv[1]), atoi(argv[2]), atoi(argv[3]));
  BgSetNumCommThread(atoi(argv[4]));
  BgSetNumWorkThread(atoi(argv[5]));

}

void BgNodeStart(int argc, char **argv) 
{
  int x, y, z;
  BgGetMyXYZ(&x, &y, &z);

  /* register handlers */
  reduceID 	   = BgRegisterHandler(reduce);
  computeMaxID = BgRegisterHandler(computeMax);
  contributeID = BgRegisterHandler(contribute);
  //CmiPrintf("BgNodeStart [%d,%d,%d]\n", x, y, z);

  //declare and set node private data
  BnvInitialize(nodeData, data);
  BnvInitialize(int, count);
  BnvAccess(count) = 0;
  for(int i=0; i<A_SIZE; i++)
    BnvAccess(data)[i] = 0 ;

  /* start computation, by triggering a micro-task at each node */
  computeMaxMsg *msg = new computeMaxMsg;
  BgSendLocalPacket(ANYTHREAD, computeMaxID, LARGE_WORK, sizeof(msg), (char*)msg);
}

/* Handler: computeMax
 * Compute the max of the numbers at my node and contribute
 */
void computeMax(char *info) 
{
  int A[A_SIZE][A_SIZE];
  int i, j;
  int max = 0;

  int x,y,z;
  BgGetMyXYZ(&x, &y, &z);


  int X, Y, Z;
  BgGetSize(&X, &Y, &Z);

  /* Initialize local data */
  for (i=0;i<A_SIZE;i++) 
    for (j=0;j<A_SIZE;j++) 
      A[i][j] = X*Y*Z - x*y*z - i*j;
      /* A[i][j] = i*j; */

  /* Find Max */
  for (i=0;i<A_SIZE;i++) 
    for (j=0;j<A_SIZE;j++) 
      if (max < A[i][j]) 
  		max = A[i][j];

  /* contribute the results for reduction : reusing the same message for optimization */
  computeMaxMsg *msg = (computeMaxMsg*)info;
  msg->max = max;
  //CmiPrintf("computeMax in [%d,%d,%d]; contributing max value %d\n", x, y, z, max);
  BgSendLocalPacket(ANYTHREAD, contributeID, SMALL_WORK, sizeof(computeMaxMsg), (char*)msg);
}

/* Handler: contribute
 * Compute the max of value contributed by current thread and previous value stored
 */
void contribute(char *info)
{
  int x,y,z;
  BgGetMyXYZ(&x, &y, &z);

  computeMaxMsg *msg = (computeMaxMsg*)info;
  //CmiPrintf("contribute in [%d,%d,%d]; reducing max value %d\n", x, y, z, msg->max);

  int reqCount;
  BnvAccess(data)[BnvAccess(count)] = msg->max;
  BnvAccess(count)++; 

  int X, Y, Z;
  BgGetSize(&X, &Y, &Z);

  if(z==Z-1) 	
  	reqCount = 1;  
  else if(z>0 || (z==0 && x==X-1))
  	reqCount = 2;
  else if(x>0 || (x==0 && y==Y-1))
  	reqCount = 3;  
  else
    reqCount = 4;

  //CmiPrintf("contribute in [%d,%d,%d]; reducing max value %d; count = %d, reqCount = %d\n", x, y, z, msg->max, ud->count, reqCount);

  if(BnvAccess(count)==reqCount)  /* if data for reduction is ready */
  {
  	//CmiPrintf("contribute in [%d,%d,%d]; reducing max value %d\n", x, y, z, msg->max);
  	BgSendLocalPacket(ANYTHREAD, reduceID, SMALL_WORK, sizeof(computeMaxMsg), (char*)msg);
    return ;
  }

}

/* Handler: reduce
 * Reduction of max values at each node to node[0,0,0]
 */
void reduce(char *info) 
{
  int x,y,z;
  BgGetMyXYZ(&x, &y, &z);

  /* do reduction */
  int max = 0;
  int count = BnvAccess(count);

  for(int i=0; i<count; i++)
  	if(max<BnvAccess(data)[i])
    	max = BnvAccess(data)[i] ; 

  if(x==0 && y==0 && z==0)
  {
  	delete (computeMaxMsg*)info;
  	//CmiPrintf("reduce in [%d,%d,%d]; Final max value %d\n", x, y, z, max);
  	CmiPrintf("Final max value %d\n", max);
  	CmiPrintf("Doing shutdown\n");
  	BgShutdown();
  	return ;
  }

  computeMaxMsg *msg = (computeMaxMsg*)info;
  msg->max = max;
  //CmiPrintf("reduce in [%d,%d,%d]; contributing max value %d\n", x, y, z, max);

  /* send max to destination(decide)  */
  if(z>0)
    z-- ;   
  else if(x>0)
    x-- ;
  else
    y-- ;

  BgSendPacket(x, y, z, ANYTHREAD, contributeID, SMALL_WORK, sizeof(computeMaxMsg), (char*)msg);
}

