/**
 * graph_coloring.C
 * Authors: Jason Cho, Laith Barakat, Garret Kiel, Philip Miller
 **/

#include <string>
#include <iostream>
#include <fstream>
#include <boost/graph/graph_traits.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/filtered_graph.hpp> //Slight problem with adding filtered graph
#include <stack>
#include <vector>
#include <queue>
#include <utility>
#include <set>
#include "charm++.h"
#include "pup_stl.h"
#include <limits>

extern "C" {
   #include "metis.h"
}

using namespace boost;
using namespace std;

#define PARALLEL
#define DSATUR

struct partition_v
{
   int num; //vertex number
   int part;
};

struct partition_e
{
   int num; //edge num. I don't intend on using this
   int part; //-1 if not part of any partition
};



typedef adjacency_list<vecS, vecS, undirectedS, 
                       partition_v, partition_e> Graph; 
typedef vector<char> Coloring;
typedef vector<char> RemVertVec;
const char UNCOLORED = -1;

typedef adjacency_iterator_generator<Graph, 
                                     graph_traits<Graph>::vertex_descriptor,
                                     graph_traits<Graph>::out_edge_iterator> adjacency_iterator;
typedef pair<graph_traits<Graph>::adjacency_iterator,
             graph_traits<Graph>::adjacency_iterator> AdjIter;

typedef graph_traits<Graph>::edge_descriptor Edge;
typedef graph_traits<Graph>::edge_iterator EdgeI;

typedef graph_traits<Graph>::vertex_descriptor Vertex;
typedef graph_traits<Graph>::vertex_iterator VertexI;

typedef property_map<Graph, int partition_v::*>::type VertMap;         
typedef property_map<Graph, int partition_e::*>::type EdgeMap;

//typedef filtered_graph< Graph, partitioner_e< EdgeMap >, partitioner_v< VertMap > >::type FilteredGraph;

template <typename VertMap>
struct partitioner_v {

   partitioner_v(){}
   partitioner_v(VertMap m,int p) : map(m), part(p)//p is partition 
   {
   }
   
   //intention : if V's partition is same as assigned,
   //we are in same partition so return true
   template <typename Vert>
   bool operator()(Vert & v) const {
      int thisPart = get(map, v);
      return (thisPart == part); 
   }
   VertMap map;
   int part;
};

template <typename EdgeMap>
struct partitioner_e {

   partitioner_e(){}
   
   partitioner_e(EdgeMap m, int p) : map(m), part(p)
   {
   }

   template <typename Ed>
   bool operator()(const Ed & e) const {
      int thisPart = get(map, e);
      return (thisPart == part);
   }

   EdgeMap map;
   int part;

};

typedef boost::filter_iterator<partitioner_v< VertMap > , VertexI> FilterVertI;
typedef boost::filter_iterator<partitioner_e< EdgeMap > , EdgeI> FilterEdgeI;


PUP::er & operator|(PUP::er &p, Graph &g)
{
    int numVerts;
    CkVec<pair<int, int> > edgeBuf;

    if (p.isPacking() || p.isSizing())
    {
        numVerts = num_vertices(g);
        int numEdges = num_edges(g);
        edgeBuf.resize(numEdges);
        EdgeI ebegin, eend;
        tie(ebegin, eend) = edges(g);
        for (int i = 0; i < numEdges; ++i)
        {
            Edge e = *ebegin++;
            pair<int, int> pr(source(e,g), target(e,g));
            edgeBuf[i] = pr;
        }
        p | numVerts;
        p | edgeBuf;
    }

    else // unpacking
    {
        p | numVerts;
        p | edgeBuf;
        Graph h(numVerts);
        for (unsigned int i = 0; i < edgeBuf.size(); ++i)
        {
            pair<int, int> pr = edgeBuf[i];
            add_edge(pr.first, pr.second, h); 
        }
        g.swap(h);
    }

    return p;
}

template<typename T, typename U>
PUP::er& operator|(PUP::er &p, pair<T, U> &pr)
{
    p | pr.first;
    p | pr.second;
  
    return p;
}

#include "graph_coloring.decl.h"

class Reportable : public CBase_Reportable
{
public:
  virtual void reportResults(Coloring &colors, int nodeNumber) = 0;
  virtual void reportNoResults(int nodeNumber) = 0;
};

/** This is the main chare that will direct how it does stuff. 
    One could think of this class as 'Graph' object, since
    there is only one graph**/
class Main : public CBase_Main
{
    const static unsigned concurrentSearches = 3;

public:
    /** We will ignore m for now **/
    Main(CkArgMsg * m)
        : maxColorsTried(0), minColor(numeric_limits<int>::max())
        {
            if (m->argc < 2 || m->argc > 3)
            {
                cout << "Incorrect invocation: graph_coloring inputfile [colors]" << endl;
                CkExit();
            }
            filename = m->argv[1];
         
            makeGraph();

	    if (m->argc == 3)
	    {
		maxColorsTried = atoi(m->argv[2]) - 1;
		startNextSearch();
	    }
	    else
	    {
		for (int i = 0; i < concurrentSearches; ++i)
		    startNextSearch();
	    }

            delete m;
        }

  void makeGraph()
  {
    ifstream infile(filename.c_str());
    Graph h;
    int numVert, numEdge;
    string s;
    int a, b;

    if (!infile)
    {
	cout << "Failed to open " << filename << endl;
	CkExit();
    }

    while(!infile.eof())
      {
	switch(infile.get())
	  {
	  case 'c': infile.ignore(100, '\n'); break;
	  case 'p': 
	    infile >> s >> numVert >> numEdge;
	    if (s != "edge")
	      {
		cout << "Unrecognized input format" << endl;
		CkExit();
	      }
	    h = Graph(numVert);
	    cout << "Parametrized: n=" << numVert << ", e=" << numEdge << endl;
	    break;
	  case 'e':
	    infile >> a >> b;
//	    cout << "Adding edge " << a << " -- " << b << endl;
	    add_edge(a-1, b-1, h);
	    break;
	  default:
	    break;
	  }
      }

    if (num_vertices(h) != numVert)
    {
	cout << "Expected " << numVert << "vertices, read " << num_vertices(h) << endl;
	CkExit();
    }

    if (num_edges(h) != numEdge)
      {
	cout << "Expected " << numEdge << " edges, read " << num_edges(h) << endl;
	CkExit();
      }

    g.swap(h);
  }

      void partitioner(Graph & h)
      {
         createVL(h); //created vertex list.
      
         detNumPart(); //determine number of partitions.
         int vertexNum = num_vertices(h);
         int wgtflag = 0;
         int numflag = 0;
         int nparts = 4;//2*vertexNum/3; //say it's 3 for now. Number of partitions.
         int options[1];
         options[0] = 0; //default option
         int edgecut;
         

         //Metis doesn't guarantee to have nparts partitions.
         //However it comes pretty close to that.         
         METIS_PartGraphKway(&vertexNum, xadj, adjncy, NULL, NULL,
            &wgtflag, &numflag, &nparts, options, &edgecut, part);
      
         //It doesn't exactly cut into nparts... that's the max it'll cut.` 
         CkPrintf("I will cut into %d partitions with %d edges\n",nparts, edgecut); 
         assignPartition(h); 
         createFGraph(h,nparts); //creates filtered graph
      }

      
      void assignPartition(Graph & h)
      {
         /** First assign partition to vertices **/
         VertexI vi;
         int count = 0;
         
         for(vi = vertices(h).first; vi != vertices(h).second; ++vi)
         {
            h[*vi].num  = count;         //assign vertex num
            h[*vi].part = part[count++]; //assigns which partition vertex is at
            CkPrintf("My partitoin : %d\n",h[*vi].part);
         }
         
         /** Then assign partition to edges. **/
         EdgeI eib, eie; //begin,end
         pair<EdgeI, EdgeI> myIter = edges(h);
         
         for(eib = myIter.first; eib != myIter.second; ++eib)
         {
            Vertex s = source(*eib,h);
            Vertex t = target(*eib,h);

            if(h[s].part == h[t].part)
               h[*eib].part = h[s].part; //connecting vertex in same partition
            else h[*eib].part = -1; //vertex not in same partition. Edge no part. 
         }
      }


      //npart refers to number of partitions
      void createFGraph(Graph & h, int nparts) {
        

         //Works needed : I can't figure out where to put filtered graph.
         //I want to put into vector, but I'm not sure how
         for(int i = 0; i < nparts; i++) {
            //second param refers to which partition of the graph to goto
            partitioner_v<VertMap> pv(get(&partition_v::part, h),i); //partition vertex
            partitioner_e<EdgeMap> ev(get(&partition_e::part, h),i); //loop through iter. 
            //This code is working, but I should put into vector, where
            //each indices refer to different partition.
            //It doesn't seem to work :-(
            //However since properties are saved, one could think of 
            //partitioning them AFTER the chares have fired them.
            filtered_graph<Graph, partitioner_e< EdgeMap >, partitioner_v< VertMap > > farg(h, ev, pv); 

            //fg.push_back(farg);
         
            //FilterVertI, FilterEdgeI needs std::pair<> to be done to iterate.
            //FilterEdgeI doesn't work. Hope that's not a problem...
            std::pair<FilterVertI, FilterVertI> va = vertices(farg); 
           
            CkPrintf("========================================\n");
            for(FilterVertI start=va.first; start != va.second; ++start)
            {
               CkPrintf("I am %d and my partition : %d\n", 
                        farg[*start].num, farg[*start].part);   

               //Only the following AdjIter is supported.
               //can't do adjacent_vertices(vertex_descriptor, farg) yet.
               //I hope that's not a problem though.
               AdjIter asdf = adjacent_vertices(farg[*start].num, h);
            }
            //fg.at(i) = farg; //<--intend to assign to vector.
//         CProxy_Node::ckNew(, thisProxy, 5, i);
         }
      
      }
      //determines number of partition to make.
      void detNumPart()
      {
         //make me!
      }


      /**creates vertex list **/
      void createVL(Graph & h)
      {
         int vertexNum = num_vertices(h);
         int edgeNum = num_edges(h);
         
         vertList.resize(vertexNum);
         AdjIter adjIter;
         xadj = new int[vertexNum+1];
         xadj[0] = 0;
         adjncy = new int[edgeNum * 2];
         part  =new int[vertexNum];
         
         int count = 0;
         for(int i = 0; i < vertexNum; i++)
         {
            
            int deg = degree(i,h);
            xadj[i+1] = xadj[i] + deg;
               
            for(adjIter = adjacent_vertices(i,h); 
               adjIter.first != adjIter.second; 
               adjIter.first++)
            {
               vertList[i].push_back(*adjIter.first); //put neighbor vert
               adjncy[count++] = *adjIter.first;
            }
         }
         
         //by now, we have safely created adjncy,xadj, and vertList.
         //vertList is used to input file,
         //xadj, adjncy is used for metis.
         
      }


    virtual void reportResults(Coloring &colors, int colorsTried)
        {
            outstanding.erase(colorsTried);
            cout << "Success with " << colorsTried << endl;
            
            if(colorsTried < minColor){
               minColor = colorsTried;
               minSuccess=colors;
            }
            
           checkFinished();
        }

    virtual void reportNoResults(int colorsTried)
        {
            cout << "Coloring failed with " << colorsTried << endl;
            outstanding.erase(colorsTried);
            checkFinished();
            startNextSearch();
        }
        
    void checkFinished()
        {
            if(outstanding.empty() || minColor <  *outstanding.begin())
            {
		if (minSuccess.size() == 0)
		{
		    cout << "Could not color " << filename << " with " << maxColorsTried << " colors." << endl;
		    CkExit();
		}

                cout << "Colors needed: " << minColor << endl;
                for(unsigned int i = 0; i < minSuccess.size(); i++)
		{
                    CkPrintf("Vertex %d color : %d\n",i,minSuccess[i]);
		}
    
		cout << "Validation says " << (validateColoring(minSuccess) ? "sucesss" : "failure") << endl;
		cout << "Colored " << filename << " on " << CkNumPes() << " PEs" << endl;

		CkExit();
            }
        }
      
private:

    void startNextSearch()
        {
	    ++maxColorsTried;
	         outstanding.insert(maxColorsTried);
            CProxy_Node::ckNew(g, thisProxy, maxColorsTried, maxColorsTried);


        }


    bool validateColoring(Coloring &colors) const
        {
            VertexI i, vend;
	    bool retval = true;
            tie(i, vend) = vertices(g);
            for(; i != vend; ++i)
            {
                if(colors[*i] == UNCOLORED)
                    return false;
                graph_traits<Graph>::adjacency_iterator j, nend;
                tie(j, nend) = adjacent_vertices(*i, g);
                for(; j != nend; ++j)
                    if(colors[*i] == colors[*j])
		    {
			cerr << "Vertices " << *i << " " << *j << " should differ" << endl;
                        retval = false;
		    }
            }

            return retval;
        }

    std::queue<int> queue;
    CProxy_Node node;
    string filename;
    Graph g;
    int maxColorsTried;
    
    
    int * part; //This contain how it is partitioned.
    int * xadj; //contains number of neighbor + prev.num.neighbor.
    int * adjncy; //contains adjncy list. Size is 2 *  edgeNum.
    vector< vector < int > > vertList;
    set<int> outstanding; //stores number of colors attempted.
    int minColor;
    Coloring minSuccess;
    
    //vector<Graph> fg;
    //vector< filtered_graph<Graph, partitioner_e< EdgeMap >, partitioner_v< VertMap > > > fg;
};



/** The node consists of several vertices, as dictated in section2.
    For sequential version however, graph == node
**/
class Node : public CBase_Node
{
public :
    Node(Graph g_, CProxy_Reportable main_, unsigned maxColors_, int nodeNumber_)
        : g(g_), vertNum(num_vertices(g)), mainProxy(main_), colors(vertNum, UNCOLORED), numCol(maxColors_), 
	  remVert(vertNum, false), nodeNumber(nodeNumber_), done(false), children(0), reported(0), success(false), steps(0)
        {
            cout << numCol << " color search started" << endl;
	    children.push_back(thisProxy);

            pair<graph_traits<Graph>::adjacency_iterator,
                graph_traits<Graph>::adjacency_iterator> adjIter;

#if 0
            cout << "Num vertices : " << num_vertices(g) << endl;
            for(int i = 0; i < num_vertices(g); i++)
            {
                cout << "Vertex " << i << " has degree: " << degree(i, g) << endl;
                cout << "Edges" << endl;
            
                /** Should make it so that it counts every adj vertex!**/
                int j = 0;
                for(adjIter = adjacent_vertices(i,g); 
                    adjIter.first != adjIter.second; 
                    adjIter.first++)
                {
                    printf("edge : %d,%d\n",i,*adjIter.first);
                    j++;
                }

                printf("Count : %d\n\n",j);

            }

            for(int i = 0; i < vertNum; i++)
            {
                CkPrintf("Vertex %d color : %d\n",i,colors[i]);
            }
#endif
            if(1 == numCol)
            {
                if(0 == num_edges(g))
                    mainProxy.reportResults(Coloring(vertNum, 0), nodeNumber);
                else
                    fail();

                return;
            }
			
			//Color highest degree node and one neighbor
			int maxDegree = -1;
			int toColor = 0;
			for( int i = 0; i < vertNum; i++ )
            {
				int curdegree = degree( i, g );
				if( curdegree > maxDegree )
				{
					toColor = i;
					maxDegree = curdegree;
				}
            }
			colors[toColor] = 0;
			
			Vertex colored = vertex( toColor, g );
			graph_traits<Graph>::adjacency_iterator j, nend;
			tie(j, nend) = adjacent_vertices(colored, g);
			maxDegree = -1;
			toColor = 0;
			for(; j != nend; ++j)
			{
				int curdegree = degree( *j, g );
				if( curdegree > maxDegree )
				{
					maxDegree = curdegree;
					toColor = *j;
				}
			}
			colors[toColor] = 1;

	    RunSearch();
        }

    Node(Graph g_, CProxy_Reportable main_, unsigned maxColors_, int nodeNumber_, Coloring col /*, RemVertVec remVerts */)
        : g(g_), vertNum(num_vertices(g)), mainProxy(main_), colors(col), numCol(maxColors_), 
	  remVert(vertNum, false), nodeNumber(nodeNumber_), done(false), children(0), reported(0), success(false), steps(0)
	{
	    //CkPrintf("Recursive child node constructor\n");
	    children.push_back(thisProxy);
	    RunSearch();
	}

    Node(CkMigrateMessage * m){}
      
    void RunSearch()
	{
            removeVertices();
            search();
            if (!success)
                fail();
	}

    virtual void reportResults(Coloring &c, int childNumber)
	{
	    // report result up to parent
	    mainProxy.reportResults(c, nodeNumber);

	    // Ensure we don't later report failure
	    success = true;

	    haltWork();
	}

    virtual void reportNoResults(int childNumber)
	{
	    ++reported;
	    // if none left, and we're done and failed, report no results to parent
	    if (/*done && */!success && children.size() == reported)
	    {
		mainProxy.reportNoResults(nodeNumber);
		haltWork();
	    }
	}

   void haltWork()
	{
	    done = true;
	    haltChildren();
	    clearData();
	}

private:


    void haltChildren()
	{
	    for (vector<CProxy_Node>::iterator i = children.begin(); i != children.end(); ++i)
		i->haltWork();
	}

    void clearData()
	{
	    g.clear();
	    colors.clear();
	    remVert.clear();
	}

    void succeed()
	{
	    success = true;
	    restoreVertices();
	    reportResults(colors, 0);
	}

    void fail()
	{
	    done = true;
	    reportNoResults(0);
	    clearData();
	}

    /** pick a vertex, and pick a color for it, and recurse **/
    void search()
        {
	    if (done)
		return;

            /**selects which vertex to start coloring **/
            int toColor = selectVert();

            if (toColor == -1)
	    {
                succeed();
		return;
	    }

            vector<bool> availableCol = availableColors(toColor);

            for(unsigned int j = 0; j < numCol && !done; j++)
            {
                if(availableCol[j])
                {
                    colors[toColor] = j;
                    if(!neighborsPossible(toColor))
                        continue;

#ifndef PARALLEL
                    search();
#else
                    ++steps;
		    if (steps % 1024 == 0)
		    {
                        spawnChild();
		    }
#if 0
                    else if (steps % 128 == 0)
                    {
                        stopped = true;
			thisProxy.RunSearch();
			return;
                    }
#endif
                    else
                        search();
#endif
                }
            }

            colors[toColor] = UNCOLORED;
        }

    void spawnChild()
	{
	  children.push_back(CProxy_Node::ckNew(g, thisProxy, numCol, children.size(), colors));
	}

    int selectVert()
        {
#ifndef DSATUR
            int toColor = -1;
            for(int i = 0; i < vertNum; i++)
            {
                // Don't try to color 'removed' vertices now
                if(remVert[i])
                    continue;
                if(colors[i] == UNCOLORED)
                {
                    toColor = i;
                    break;
                }
            }
#else
            int toColor = -1;
            int maxDegree = -1;
            int maxSatDegree = -1;
            for( int i = 0; i < vertNum; i++ )
            {
                if(remVert[i])
                    continue;
                if(colors[i] == UNCOLORED)
                {
                    Vertex vert = vertex( i, g );
                    set<int> neighborColors;
                    graph_traits<Graph>::adjacency_iterator j, nend;
			        tie(j, nend) = adjacent_vertices(vert, g);
			        for(; j != nend; ++j)
			        {
                        if( colors[*j] != UNCOLORED )
				            neighborColors.insert(colors[*j]);
			        }

                    int size = neighborColors.size();
                    //Choose the vertex with the maximum saturation degree
                    if( size > maxSatDegree )
                    {
                        maxSatDegree = size;
                        maxDegree = degree( vert, g );
                        toColor = vert;
                    }
                    //If vertices tie for max saturation degree, take the one with the larger degree
                    else if( size == maxSatDegree )
                    {
                        int newDegree = degree( vert, g );
                        int bigger = ( maxDegree >= newDegree ) ? toColor : vert;
                        if( bigger == vert )
                        {
                            maxDegree = newDegree;
                            toColor = vert;
                        }
                    }
                }
            }

            //Find vertex with max saturation
#endif
            return toColor;
        }


    void removeVertices()
        {
            bool changed;
            do {
                changed = false;
                for(int i = 0; i < vertNum; i++)
                {
                    if(!remVert[i] && degree(i,g) < numCol)
                    {
                        changed = true;
                        pair<int, vector<int> > toRem;
                        toRem.first = i;
                        AdjIter adjIter = adjacent_vertices(i, g);

                        toRem.second.insert(toRem.second.begin(),
                                            adjIter.first, adjIter.second);
                        vertStack.push_back(toRem);

                        clear_vertex(i, g);
                        remVert[i] = true;
                    }
                }
            } while (changed);
        }

    void restoreVertices()
        {
   
            /** removing vertices **/
            while(!vertStack.empty())
            {
                int vertRestore = vertStack[vertStack.size()-1].first;
                vector<int> edgeRestore = vertStack[vertStack.size()-1].second;
                for (vector<int>::iterator i = edgeRestore.begin(); i != edgeRestore.end(); ++i)
                    add_edge(vertRestore, *i, g);
                vertStack.pop_back();
                remVert[vertRestore] = false;
         
                //color this vertex.
                vector<bool> availCol = availableColors(vertRestore);
         
                for(unsigned  int i = 0; i < availCol.size(); i++)
                    if(availCol[i])
                    {
                        colors[vertRestore] = i;
                        break;
                    }
            }
     
     
        }


    bool neighborsPossible(int justColored)
        {
            // Find adjacent vertices
            for(AdjIter adjIter = adjacent_vertices(justColored, g);
                adjIter.first != adjIter.second;
                adjIter.first++)
            {
                bool possible = false;
                //   check for available color
                vector<bool> avail = availableColors(*(adjIter.first));
                //   if none available, return false
                for(unsigned int i = 0; i < numCol; ++i)
                    if(avail[i])
                    {
                        possible = true;
                        break;
                    }
                if(!possible)
                    return false;
            }

            return true;
        }

    vector<bool> availableColors(int v)
        {
            AdjIter adjIter;
            vector<bool> avail(numCol, true);
    
            for(adjIter = adjacent_vertices(v, g); 
                adjIter.first != adjIter.second;
                adjIter.first++)
            {
                int color = colors[*(adjIter.first)];
                if (UNCOLORED != color)
                    avail[color] = false;
            }
    
            return avail;
        }
  
    Graph g;
    int vertNum;
    unsigned int numCol;
    CProxy_Reportable mainProxy;
    Coloring colors;
    vector<pair<int, vector<int> > > vertStack;
    RemVertVec remVert;
    int nodeNumber;
    bool done;
    int reported;
    vector<CProxy_Node> children;
    bool success;
    int steps;
};

#include "graph_coloring.def.h"
